#pragma once

#include "base/Geometry/HorizontalRange.h"
#include <limits>

inline void
InitialiseRangeToIntegerLimits(cHorizontalRange& r)
{
    r.minX = r.minY = std::numeric_limits<int32_t>::max();
    r.maxX = r.maxY = std::numeric_limits<int32_t>::min();
}

inline void
InitialiseRangeToMax(cHorizontalRange& r)
{
    r.minX = r.minY = std::numeric_limits<int32_t>::min();
    r.maxX = r.maxY = std::numeric_limits<int32_t>::max();
}

inline void
InitialiseRange(cHorizontalRange& r, int32_t x, int32_t y)
{
    r.minX = r.maxX = x;
    r.minY = r.maxY = y;
}

inline void
ExtendRange(cHorizontalRange& r, int32_t x, int32_t y)
{
    if(x < r.minX)
    {
        r.minX = x;
    }
    if(x > r.maxX)
    {
        r.maxX = x;
    }
    if(y < r.minY)
    {
        r.minY = y;
    }
    if(y > r.maxY)
    {
        r.maxY = y;
    }
}

inline void
ExtendRangeBy(cHorizontalRange& r, int32_t amount)
{
    r.minX -= amount;
    r.maxX += amount;
    r.minY -= amount;
    r.maxY += amount;
}

inline bool
RangesTouchOrOverlap(const cHorizontalRange& r1, const cHorizontalRange& r2)
{
    if(r1.maxX < r2.minX)
    {
        return false;
    }
    if(r1.minX > r2.maxX)
    {
        return false;
    }
    if(r1.maxY < r2.minY)
    {
        return false;
    }
    if(r1.minY > r2.maxY)
    {
        return false;
    }
    return true;
}
inline bool
RangesOverlap(const cHorizontalRange& r1, const cHorizontalRange& r2)
{
    if(r1.maxX <= r2.minX)
    {
        return false;
    }
    if(r1.minX >= r2.maxX)
    {
        return false;
    }
    if(r1.maxY <= r2.minY)
    {
        return false;
    }
    if(r1.minY >= r2.maxY)
    {
        return false;
    }
    return true;
}

inline bool
RangeContainsPoint_Inclusive(const cHorizontalRange& r, int32_t x, int32_t y)
{
    return x >= r.minX
        && x <= r.maxX
        && y >= r.minY
        && y <= r.maxY
        ;
}
inline bool
RangeContainsPoint_NonInclusive(const cHorizontalRange& r, int32_t x, int32_t y)
{
    return x > r.minX
        && x < r.maxX
        && y > r.minY
        && y < r.maxY
        ;
}

inline bool
RangeContainsRange(const cHorizontalRange& lhs, const cHorizontalRange& rhs)
{
    return RangeContainsPoint_Inclusive(lhs, rhs.minX, rhs.minY)
        && RangeContainsPoint_Inclusive(lhs, rhs.maxX, rhs.maxY)
        ;
}

inline int32_t
CoordinateRangeUsed(const cHorizontalRange& r)
{
    int32_t result = r.minX;
    if(result < 0)
    {
        result = -result;
    }
    int32_t candidate;
    candidate = r.minY;
    if(candidate < 0)
    {
        candidate = -candidate;
    }
    if(candidate > result)
    {
        result = candidate;
    }
    candidate = r.maxX;
    if(candidate < 0)
    {
        candidate = -candidate;
    }
    if(candidate > result)
    {
        result = candidate;
    }
    candidate = r.maxY;
    if(candidate < 0)
    {
        candidate = -candidate;
    }
    if(candidate > result)
    {
        result = candidate;
    }
    return result;
}

inline void
AddRange(cHorizontalRange& addTo, const cHorizontalRange& toAdd)
{
    if(toAdd.minX < addTo.minX)
    {
        addTo.minX = toAdd.minX;
    }
    if(toAdd.minY < addTo.minY)
    {
        addTo.minY = toAdd.minY;
    }
    if(toAdd.maxX > addTo.maxX)
    {
        addTo.maxX = toAdd.maxX;
    }
    if(toAdd.maxY > addTo.maxY)
    {
        addTo.maxY = toAdd.maxY;
    }
}

inline void
GetRangeIntersection_KnownToIntersect(cHorizontalRange& dest, const cHorizontalRange& src)
{
    assertD(RangesTouchOrOverlap(src, dest));
    if(src.minX > dest.minX)
    {
        dest.minX = src.minX;
    }
    if(src.minY > dest.minY)
    {
        dest.minY = src.minY;
    }
    if(src.maxX < dest.maxX)
    {
        dest.maxX = src.maxX;
    }
    if(src.maxY < dest.maxY)
    {
        dest.maxY = src.maxY;
    }
}

inline int32_t&
IndexedRangeMin(cHorizontalRange& range, int index)
{
    if(index == 0)
    {
        return range.minX;
    }
    return range.minY;
}
inline int32_t&
IndexedRangeMax(cHorizontalRange& range, int index)
{
    if(index == 0)
    {
        return range.maxX;
    }
    return range.maxY;
}
inline const int32_t&
IndexedRangeMin(const cHorizontalRange& range, int index)
{
    if(index == 0)
    {
        return range.minX;
    }
    return range.minY;
}
inline const int32_t&
IndexedRangeMax(const cHorizontalRange& range, int index)
{
    if(index == 0)
    {
        return range.maxX;
    }
    return range.maxY;
}

// ** note that there are no range checks here **
// ** calling code needs to ensure that there is no possibility of overflow **
inline void
TranslateRange(cHorizontalRange& range, int32_t x, int32_t y)
{
    range.minX += x;
    range.minY += y;
    range.maxX += x;
    range.maxY += y;
}
