//**********************************************************************
//
// Copyright (c) 2002-2019
// Thomas Young, PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#ifndef PATHENGINE_INTERFACE_HAS_BEEN_INCLUDED
#define PATHENGINE_INTERFACE_HAS_BEEN_INCLUDED

#include <vector>
#include <string>
#include <cstdint>
#include <memory>

#ifndef PATHENGINE_POD_CLASSES_HAVE_BEEN_DEFINED
#define PATHENGINE_POD_CLASSES_HAVE_BEEN_DEFINED
class cPosition
{
public:
    cPosition() : cell(-1) {}
    cPosition(int32_t x, int32_t y, int32_t cell) : x(x), y(y), cell(cell) {}
    int32_t x, y, cell;
    bool operator==(const cPosition& rhs) const
    {
        if(cell == -1)
            return cell == rhs.cell;
        return cell == rhs.cell && x == rhs.x && y == rhs.y;
    }
    bool operator!=(const cPosition& rhs) const
    {
        return !((*this) == rhs);
    }
};
class cHorizontalRange
{
public:
    int32_t minX, minY, maxX, maxY;
    bool operator==(const cHorizontalRange& rhs) const
    {
        return minX == rhs.minX && minY == rhs.minY && maxX == rhs.maxX && maxY == rhs.maxY;
    }
    bool operator!=(const cHorizontalRange& rhs) const
    {
        return !((*this) == rhs);
    }
};
class cCollidingLine
{
public:
    int32_t startX, startY, endX, endY;
};
#endif

#ifndef PATHENGINE_iArrayReceiver_DEFINED
#define PATHENGINE_iArrayReceiver_DEFINED
template <class T> class iArrayReceiver
{
public:
    virtual ~iArrayReceiver() {}
    virtual void receive(const T& element) = 0;
};
#endif
#ifndef PATHENGINE_cVectorReceiver_DEFINED
#define PATHENGINE_cVectorReceiver_DEFINED
template <class T> class cVectorReceiver : public iArrayReceiver<T>
{
public:
    std::vector<T> vector;
    void
    receive(const T& element)
    {
        vector.push_back(element);
    }
    T*
    buffer()
    {
        return vector.empty() ? 0 : &vector.front();
    }
    uint32_t
    bufferElements() const
    {
        return static_cast<uint32_t>(vector.size());
    }
};
#endif
#ifndef PATHENGINE_cAttributesReceiver_DEFINED
#define PATHENGINE_cAttributesReceiver_DEFINED
 class cStringVectorReceiver : public iArrayReceiver<const char*>
{
public:
    std::vector<std::string> vector;

    void
    receive(const char* const& element)
    {
        vector.push_back(element);
    }
};
#endif
#ifndef PATHENGINE_cAPIRefReceiver_DEFINED
#define PATHENGINE_cAPIRefReceiver_DEFINED
template <class tVector> class cAPIRefReceiver : public iArrayReceiver<typename tVector::value_type::element_type*>
{
    tVector* _v;
public:
    cAPIRefReceiver(tVector& v)
    {
        _v = &v;
    }
    void
    receive(typename tVector::value_type::element_type *const& element)
    {
        _v->emplace_back(element);
    }
};
#endif

class iOwnershipGraphRoot;
class iExpanded2DObstacles;
class iFaceVertexMesh;
class iFaceVertexBuilder;
class iSolidObjectsBuilder;
class iOutputStream;
class iAgent;
class iGround;
class iMesh;
class iPathEngine;
class iShape;
class iCollisionContext;
class iObstacleSet;
class iPath;
class iTerrainCallBack;
class iRender3DLinesCallBack;
class iQueryCallBack;
class iContentChunk;
class iContentChunkInstance;
class iMeshFederation;
class iSourceTiling;
class iAnchorsAndPinnedShapes;
class iSolidObjects;
class iMeshFederationTileGeneratedCallBack;
class iConnectionTestCallBack;
class iProgressCallBack;
class iErrorHandler;
class iContentProcessing3D;
class iTestBed;
class iRenderGeometry;

const int32_t PATHENGINE_INTERFACE_MAJOR_VERSION = 31;
const int32_t PATHENGINE_INTERFACE_MINOR_VERSION = 1;
const int32_t CONTENT_PROCESSING_3D_INTERFACE_MAJOR_VERSION = 10;
const int32_t CONTENT_PROCESSING_3D_INTERFACE_MINOR_VERSION = 0;
const int32_t TESTBED_INTERFACE_MAJOR_VERSION = 10;
const int32_t TESTBED_INTERFACE_MINOR_VERSION = 0;

#ifndef PATHENGINE_iOwnershipGraphRoot_DEFINED
#define PATHENGINE_iOwnershipGraphRoot_DEFINED
class iOwnershipGraphRoot
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iOwnershipGraphRoot* addExternalRef__Raw() = 0;
    virtual const iOwnershipGraphRoot* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iOwnershipGraphRoot> addExternalRef()
    {
        auto result = std::unique_ptr<iOwnershipGraphRoot>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iOwnershipGraphRoot> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iOwnershipGraphRoot>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iOwnershipGraphRoot* afterCast = static_cast<iOwnershipGraphRoot*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

#ifndef PATHENGINE_iExpanded2DObstacles_DEFINED
#define PATHENGINE_iExpanded2DObstacles_DEFINED
class iExpanded2DObstacles
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iExpanded2DObstacles* addExternalRef__Raw() = 0;
    virtual const iExpanded2DObstacles* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Saves the expanded obstacles to persistent storage. 
    virtual void save(iOutputStream& outputStream) const = 0;
    // A helper method for drawing the expanded and combined boundaries. 
    virtual void render(int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // Returns coordinates for the expand shape that was used to generate this object. 
    virtual void getExpandShapeCoords(iArrayReceiver<int32_t>& result) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iExpanded2DObstacles> addExternalRef()
    {
        auto result = std::unique_ptr<iExpanded2DObstacles>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iExpanded2DObstacles> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iExpanded2DObstacles>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iExpanded2DObstacles* afterCast = static_cast<iExpanded2DObstacles*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

#ifndef PATHENGINE_iFaceVertexMesh_DEFINED
#define PATHENGINE_iFaceVertexMesh_DEFINED
class iFaceVertexMesh
{
public:
    virtual ~iFaceVertexMesh() {}
    // Returns the number of faces (tris) in this object. 
    virtual int32_t faces() const = 0;
    // Returns the number of unique vertices in this object. 
    virtual int32_t vertices() const = 0;
    // Enables you to obtain the indices of vertices around a given face. 
    virtual int32_t vertexIndex(int32_t face, int32_t vertexInFace) const = 0;
    // Used to obtain vertex coordinates. 
    virtual int32_t vertexX(int32_t vertexIndex) const = 0;
    // Used to obtain vertex coordinates. 
    virtual int32_t vertexY(int32_t vertexIndex) const = 0;
    // Used to obtain vertex coordinates. 
    virtual float vertexZ(int32_t vertexIndex) const = 0;
    // Used to obtain information about a face. 
    virtual int32_t faceAttribute(int32_t face, int32_t attributeIndex) const = 0;
};
#endif

#ifndef PATHENGINE_iFaceVertexBuilder_DEFINED
#define PATHENGINE_iFaceVertexBuilder_DEFINED
class iFaceVertexBuilder
{
public:
    virtual ~iFaceVertexBuilder() {}
    // Informs the builder that calls to addVertex() and so on will follow, for a new mesh. 
    virtual void startMesh() = 0;
    // Adds a vertex to the mesh, and returns the index of the newly added vertex. 
    virtual int32_t addVertex(int32_t x, int32_t y, float z) = 0;
    // Adds a face (triangle) to the mesh, and returns the index of the newly added face. 
    virtual int32_t addFace(int32_t vertex0, int32_t vertex1, int32_t vertex2) = 0;
    // (Optionally) sets an attribute for a face in the mesh. 
    virtual void setFaceAttribute(int32_t faceIndex, int32_t attributeType, int32_t attributeValue) = 0;
};
#endif

#ifndef PATHENGINE_iSolidObjectsBuilder_DEFINED
#define PATHENGINE_iSolidObjectsBuilder_DEFINED
class iSolidObjectsBuilder
{
public:
    virtual ~iSolidObjectsBuilder() {}
    // Informs the builder that calls to addPoint() and so on will follow, for a new solid. 
    virtual void startSolid() = 0;
    // Adds a point to the current solid. 
    virtual void addPoint(int32_t x, int32_t y, int32_t z) = 0;
    // (Optionally) sets an attribute for the convex solid currently being built. 
    virtual void setAttribute(int32_t attributeType, int32_t attributeValue) = 0;
};
#endif

#ifndef PATHENGINE_iOutputStream_DEFINED
#define PATHENGINE_iOutputStream_DEFINED
class iOutputStream
{
public:
    virtual ~iOutputStream() {}
    // This method is called by PathEngine to send data to an application defined output stream. 
    virtual void put(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
};
#endif

#ifndef PATHENGINE_iAgent_DEFINED
#define PATHENGINE_iAgent_DEFINED
class iAgent
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iAgent* addExternalRef__Raw() = 0;
    virtual const iAgent* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    virtual iMesh* getMesh__Raw() const = 0;
    virtual const iShape* getShape__Raw() const = 0;
    // Enables the client application to associate arbitrary application data with an agent. 
    virtual void setUserData(int64_t value) = 0;
    // Used to retrieve a void pointer previously passed in to iAgent::setUserData(). 
    virtual int64_t getUserData() const = 0;
    // A helper function to obtain the current position of the agent. 
    virtual cPosition getPosition() const = 0;
    // Move the agent to a specified position, without any collision checking. 
    virtual void moveTo(const cPosition& newPosition) = 0;
    // A helper function to determine whether collision queries can be made for the agent. 
    virtual bool canCollide() const = 0;
    // A helper function to determine whether pathfinding queries can be made for the agent. 
    virtual bool canPathfind(const iCollisionContext* context) const = 0;
    // Tests whether the agent can be placed at a specified position. 
    virtual bool testCollisionAt(const iCollisionContext* context, const cPosition& newPosition) const = 0;
    // Tests whether the agent can move along a line to a specified position. 
    virtual bool testCollisionTo(const iCollisionContext* context, const cPosition& newPosition) const = 0;
    // Tests whether the agent can move along a line specified in terms of target x and y coordinates. 
    virtual bool testCollisionTo_XY(const iCollisionContext* context, int32_t x, int32_t y, int32_t& cell) const = 0;
    virtual bool firstCollisionTo__Raw(const iCollisionContext* context, int32_t x, int32_t y, int32_t& cell, cCollidingLine& collidingLine, iAgent*& agent) const = 0;
    // Finds the closest unobstructed position, if one exists within a specified region around the agent's position. 
    virtual cPosition findClosestUnobstructedPosition(const iCollisionContext* context, int32_t maximumDistance) const = 0;
    virtual void getAllAgentsOverlapped__Raw(const iCollisionContext* context, iArrayReceiver<iAgent*>& overlappedAgents) const = 0;
    // Tests whether a specified shape at a specified position will obstruct the agent. 
    virtual bool testCollisionDirectlyAgainstPlacedShape(const iShape& shape, const cPosition& shapePosition) const = 0;
    virtual iPath* findShortestPathTo__Raw(const iCollisionContext* context, const cPosition& target) const = 0;
    virtual iPath* findShortestPathTo_WithQueryCallBack__Raw(const iCollisionContext* context, const cPosition& target, iQueryCallBack* queryCallBack) const = 0;
    virtual iPath* findPathAway__Raw(const iCollisionContext* context, const cPosition& awayFrom, int32_t distanceAway) const = 0;
    virtual iPath* findPathAway_WithQueryCallBack__Raw(const iCollisionContext* context, const cPosition& awayFrom, int32_t distanceAway, iQueryCallBack* queryCallBack) const = 0;
    // Advances the agent along a path without allowing the agent to enter obstructed regions because of position approximation. 
    virtual bool advanceAlongPath(iPath* path, float distance, const iCollisionContext* context) = 0;
    // Similar to advanceAlongPath(), but provides extra precision components of the agent's position. 
    virtual bool advanceAlongPathWithPrecision(iPath* path, float distance, const iCollisionContext* context, float& precisionX, float& precisionY) = 0;
    virtual bool advanceAlongPathWithCollisionInfo__Raw(iPath* path, float distance, const iCollisionContext* context, float& precisionX, float& precisionY, cCollidingLine& collidingLine, iAgent*& agent) = 0;
    virtual iPath* generateCurvedPath__Raw(iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const = 0;
    // Specifies if the obstacle should function as a 'soft obstacle', and specifies the traversal cost. 
    virtual void setTraverseCost(float cost) = 0;
    // Enables a direction to be associated with the traverse cost for a soft obstacle. 
    virtual void setTraverseCostDirection(int32_t directionVectorX, int32_t directionVectorY) = 0;
    // A helper function which can be used to determine whether an agent is set to behave as a 'soft obstacle', and the traverse cost associated with the agent. 
    virtual float getTraverseCost() const = 0;
    // A helper function which can be used to determine whether a direction has been associated with a soft obstacle, and retrieve the direction. 
    virtual void getTraverseCostDirection(int32_t& directionVectorX, int32_t& directionVectorY) const = 0;
    virtual iPath* generateCurvedPath_WithEndVector__Raw(iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t endVectorX, int32_t endVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const = 0;
    // A helper function to determine whether an agent can be moved. 
    virtual bool isMoveable() const = 0;
    // A helper function for access to underlying agent geometry, e.g. for debug rendering. 
    virtual int32_t getNumberOfVertices() const = 0;
    // A helper function for access to underlying agent geometry, e.g. for debug rendering. 
    virtual void getVertexCoordinates(int32_t index, int32_t& x, int32_t& y) const = 0;
    // A helper function to access the mesh on which this agent is placed. 
    virtual iMesh& refMesh() const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iAgent> addExternalRef()
    {
        auto result = std::unique_ptr<iAgent>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iAgent> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iAgent>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iAgent* afterCast = static_cast<iAgent*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // A helper function to access the mesh on which this agent is placed. 
    std::unique_ptr<iMesh> getMesh() const 
    {
        auto result = std::unique_ptr<iMesh>(getMesh__Raw());
        return result;
    }
    // A helper function to obtain the Interface iShape that was used to place the agent. 
    std::unique_ptr<const iShape> getShape() const 
    {
        auto result = std::unique_ptr<const iShape>(getShape__Raw());
        return result;
    }
    // Determines the first contact, if any, for movement by the agent along a line to a specified target position. 
    bool firstCollisionTo(const iCollisionContext* context, int32_t x, int32_t y, int32_t& cell, cCollidingLine& collidingLine, std::unique_ptr<iAgent>& agent) const 
    {
        iAgent* agent__Raw;
        auto result = firstCollisionTo__Raw(context, x, y, cell, collidingLine, agent__Raw);
        agent = std::unique_ptr<iAgent>(agent__Raw);
        return result;
    }
    // Fills a buffer with the set of agents in a given context overlapped by this agent. 
    template <class tVector> void getAllAgentsOverlapped(const iCollisionContext* context, tVector& overlappedAgents) const 
    {
        cAPIRefReceiver<tVector> overlappedAgents__Receiver(overlappedAgents);
        getAllAgentsOverlapped__Raw(context, overlappedAgents__Receiver);
    }
    // Finds the shortest path from the agent's current position to a specified target position. 
    std::unique_ptr<iPath> findShortestPathTo(const iCollisionContext* context, const cPosition& target) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPathTo__Raw(context, target));
        return result;
    }
    // Finds the shortest path from the agent's current position to a specified target position.
    // (This version supports passing in a callback object to enable the query to be aborted early.) 
    std::unique_ptr<iPath> findShortestPathTo_WithQueryCallBack(const iCollisionContext* context, const cPosition& target, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPathTo_WithQueryCallBack__Raw(context, target, queryCallBack));
        return result;
    }
    // Finds the shortest path of escape from a specified region for the agent. 
    std::unique_ptr<iPath> findPathAway(const iCollisionContext* context, const cPosition& awayFrom, int32_t distanceAway) const 
    {
        auto result = std::unique_ptr<iPath>(findPathAway__Raw(context, awayFrom, distanceAway));
        return result;
    }
    // Finds the shortest path of escape from a specified region for the agent.
    // (This version supports passing in a callback object to enable the query to be aborted early.) 
    std::unique_ptr<iPath> findPathAway_WithQueryCallBack(const iCollisionContext* context, const cPosition& awayFrom, int32_t distanceAway, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findPathAway_WithQueryCallBack__Raw(context, awayFrom, distanceAway, queryCallBack));
        return result;
    }
    // A version of advanceAlongPath() with extra precision components of the agent's position and collision information. 
    bool advanceAlongPathWithCollisionInfo(iPath* path, float distance, const iCollisionContext* context, float& precisionX, float& precisionY, cCollidingLine& collidingLine, std::unique_ptr<iAgent>& agent)
    {
        iAgent* agent__Raw;
        auto result = advanceAlongPathWithCollisionInfo__Raw(path, distance, context, precisionX, precisionY, collidingLine, agent__Raw);
        agent = std::unique_ptr<iAgent>(agent__Raw);
        return result;
    }
    // Post process path smoothing.
    // Returns a new path based on a 'base path', but with corners smoothed according to supplied parameters. 
    std::unique_ptr<iPath> generateCurvedPath(iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const 
    {
        auto result = std::unique_ptr<iPath>(generateCurvedPath__Raw(basePath, context, startVectorX, startVectorY, sectionLength, turnRatio1, turnRatio2));
        return result;
    }
    // Post process path smoothing.
    // Returns a new path based on a 'base path', but with corners smoothed according to supplied parameters. 
    std::unique_ptr<iPath> generateCurvedPath_WithEndVector(iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t endVectorX, int32_t endVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const 
    {
        auto result = std::unique_ptr<iPath>(generateCurvedPath_WithEndVector__Raw(basePath, context, startVectorX, startVectorY, endVectorX, endVectorY, sectionLength, turnRatio1, turnRatio2));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iGround_DEFINED
#define PATHENGINE_iGround_DEFINED
class iGround
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iGround* addExternalRef__Raw() = 0;
    virtual const iGround* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual int32_t getNumberOf3DFaces() const = 0;
    // A helper function for obtaining information about the 3D mesh under a position. 
    virtual int32_t get3DFaceAtPosition(const cPosition& position) const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual void get3DFaceVertex(int32_t faceIndex, int32_t vertexIndexInFace, int32_t& x, int32_t& y, float& z) const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual int32_t get3DFaceConnection(int32_t faceIndex, int32_t edgeIndexInFace) const = 0;
    // Returns a normal vector for a face in the underlying 3d mesh.
    virtual void get3DFaceNormal(int32_t faceIndex, float& resultX, float& resultY, float& resultZ) const = 0;
    // Used to query ground attributes.
    virtual int32_t get3DFaceAttribute(int32_t faceIndex, int32_t attributeIndex) const = 0;
    // Helper function returning the number of sections present in the underlying 3D mesh.
    virtual int32_t getNumberOfSections() const = 0;
    // Queries for the surface type value associated with the given position on a ground mesh. 
    virtual int32_t getSurfaceType(const cPosition& position) const = 0;
    // Returns the section ID for a given position. 
    virtual int32_t getSectionID(const cPosition& position) const = 0;
    // Returns whether or not the specified position is a valid position on this ground. 
    virtual bool positionIsValid(const cPosition& p) const = 0;
    // Extracts a set of convex polygons that can be used as a 2D collision footprint for the horizontal extents of a ground mesh. 
    virtual void extractConvexPolyFootprint(iArrayReceiver<int32_t>& result) const = 0;
    // Saves part or all of the information in a ground object to persistence. 
    virtual void save(const char* format, bool includeMapping, bool includeTranslationTo2D, bool includePartitioning, iOutputStream& outputStream) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iGround> addExternalRef()
    {
        auto result = std::unique_ptr<iGround>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iGround> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iGround>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iGround* afterCast = static_cast<iGround*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

#ifndef PATHENGINE_iMesh_DEFINED
#define PATHENGINE_iMesh_DEFINED
class iMesh
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iMesh* addExternalRef__Raw() = 0;
    virtual const iMesh* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual int32_t getNumberOf3DFaces() const = 0;
    // A helper function for obtaining information about the 3D mesh under a position. 
    virtual int32_t get3DFaceAtPosition(const cPosition& position) const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual void get3DFaceVertex(int32_t faceIndex, int32_t vertexIndexInFace, int32_t& x, int32_t& y, float& z) const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual int32_t get3DFaceConnection(int32_t faceIndex, int32_t edgeIndexInFace) const = 0;
    // Returns a normal vector for a face in the underlying 3d mesh.
    virtual void get3DFaceNormal(int32_t faceIndex, float& resultX, float& resultY, float& resultZ) const = 0;
    // Used to query ground attributes.
    virtual int32_t get3DFaceAttribute(int32_t faceIndex, int32_t attributeIndex) const = 0;
    // Helper function returning the number of sections present in the underlying 3D mesh.
    virtual int32_t getNumberOfSections() const = 0;
    // Queries for the surface type value associated with the given position on a ground mesh. 
    virtual int32_t getSurfaceType(const cPosition& position) const = 0;
    // Returns the section ID for a given position. 
    virtual int32_t getSectionID(const cPosition& position) const = 0;
    // Intended for use when building content. Associates a position and orientation with a string ID. 
    virtual void storeAnchor(const char* id, const cPosition& position, int32_t orientation) = 0;
    // Retrieve named positions stored with this mesh. 
    virtual cPosition retrieveAnchor(const char* id, int32_t& orientation) const = 0;
    // Returns the number of anchors stored with this mesh. 
    virtual int32_t getNumberOfAnchors() const = 0;
    // Used for iterating through anchors.
    virtual void retrieveAnchorByIndex(int32_t index, cPosition& position, int32_t& orientation, const char*& id) const = 0;
    // Intended for use when building content. Places an obstacle that can later be retrieved as an agent. 
    virtual void storeNamedObstacle(const char* id, iAgent& agent) = 0;
    virtual iAgent* retrieveNamedObstacle__Raw(const char* id) const = 0;
    // Returns the number of fixed obstacles stored with this mesh. 
    virtual int32_t getNumberOfNamedObstacles() const = 0;
    virtual void retrieveNamedObstacleByIndex__Raw(int32_t index, iAgent*& agent, const char*& id) const = 0;
    // Enables you to obtain a position on a mesh underneath a point in 3D space. 
    virtual cPosition positionFor3DPoint(int32_t x, int32_t y, int32_t z) const = 0;
    // Enables you to resolve a position in 3d space to a nearby position on the surface of this ground mesh. 
    virtual cPosition positionNear3DPoint(int32_t x, int32_t y, int32_t z, int32_t horizontalRange, int32_t verticalRange) const = 0;
    // Obtains a position in the specified section and at the given coordinates, if such a position exists. 
    virtual cPosition positionInSection(int32_t sectionID, int32_t x, int32_t y) const = 0;
    // Variation on iMesh::positionFor3DPoint() enabling the resulting position to be restricted to a specified section. 
    virtual cPosition positionInSectionFor3DPoint(int32_t sectionID, int32_t x, int32_t y, int32_t z) const = 0;
    // Variation on iMesh::positionNear3DPoint() enabling the resulting position to be restricted to a specified section. 
    virtual cPosition positionInSectionNear3DPoint(int32_t sectionID, int32_t x, int32_t y, int32_t z, int32_t horizontalRange, int32_t verticalRange) const = 0;
    // Extended position resolution method that can return multiple positions when more than one piece of ground mesh is local to the query point, and also supports filtering by sectionID if this is desired. 
    virtual void resolvePositionsNear3DPoint(int32_t sectionID, int32_t x, int32_t y, int32_t z, const cHorizontalRange& horizontalQueryRegion, int32_t minimumZ, int32_t maximumZ, iArrayReceiver<cPosition>& positions) const = 0;
    // Used for resolving anchor positions in the case where the anchor should not resolve to 'terrain stand-in' geometry. 
    virtual cPosition positionFor3DPoint_ExcludeTerrain(int32_t x, int32_t y, int32_t z, int32_t numberOfTerrainLayers) const = 0;
    // Generates a random position on the surface of this mesh. 
    virtual cPosition generateRandomPosition() const = 0;
    // Generates a random position on the region of a mesh local to the specified centre position. 
    virtual cPosition generateRandomPositionLocally(const cPosition& centre, int32_t range) const = 0;
    // Generates a random position in the specified section. 
    virtual cPosition generateRandomPositionInSection(int32_t sectionID) const = 0;
    // Provides the height for a position on the surface of this ground mesh. 
    virtual int32_t heightAtPosition(const cPosition& position) const = 0;
    // Provides the height for a position on the surface of this ground mesh (as a float). 
    virtual float heightAtPositionF(const cPosition& position) const = 0;
    // Provides the height for a position on the surface of this ground mesh as a float, taking account of the precision components of the position supplied. 
    virtual float heightAtPositionWithPrecision(const cPosition& position, float precisionX, float precisionY) const = 0;
    // Traverses along the surface of a ground mesh to disambiguate overlapping geometry at a target point. 
    virtual int32_t getCellForEndOfLine(const cPosition& start, int32_t endX, int32_t endY) const = 0;
    // Returns whether or not the specified position is a valid position on this mesh. 
    virtual bool positionIsValid(const cPosition& p) const = 0;
    // Adds boundaries corresponding to the set of obstacles in a context to the mesh's base circuits.
    virtual void burnContextIntoMesh(const iCollisionContext& context) = 0;
    // Saves the underlying ground for this iMesh (backward compatibility version). 
    virtual void saveGround(const char* format, bool includeMapping, iOutputStream& outputStream) const = 0;
    // Supplies a callback object to be used by PathEngine for looking up height detail at points in the specified terrain layer on this mesh. 
    virtual void setTerrainCallBack(int32_t terrainLayer, iTerrainCallBack* callBack) = 0;
    // Adds an endpoint for use with the off-mesh connections feature. 
    virtual int32_t addEndPoint(const cPosition& position) = 0;
    // Tells PathEngine about the possibility for agents to move 'off-mesh' between a pair of endpoints. 
    virtual int32_t addOffMeshConnection(int32_t fromEndPoint, int32_t toEndPoint, int32_t penalty) = 0;
    // Helper function returning the number of off-mesh connection endpoints stored with a mesh. 
    virtual int32_t getNumberOfEndPoints() const = 0;
    // Helper function for retrieving the positions for off-mesh connection endpoints stored with a mesh. 
    virtual cPosition getEndPoint(int32_t index) const = 0;
    // Helper function returning the number of off-mesh connections stored with a mesh. 
    virtual int32_t getNumberOfOffMeshConnections() const = 0;
    // Helper function for retrieving information about off-mesh connections stored with a mesh. 
    virtual void getOffMeshConnectionInfo(int32_t index, int32_t& fromEndPoint, int32_t& toEndPoint, int32_t& penalty) const = 0;
    // Removes any off-mesh connections or off-mesh connection endpoints associated with a mesh. 
    virtual void clearOffMeshConnectionsAndEndPoints() = 0;
    // Generates a representation of (expanded) unobstructed space boundaries for the specified agent shape, on this mesh. 
    virtual void generateUnobstructedSpaceFor(const iShape& shape, bool combine, const char *const* options) = 0;
    // Generates a representation of (expanded) unobstructed space boundaries for the specified agent shape, on this mesh. 
    virtual void generateUnobstructedSpaceFor_WithPreExpanded(const iShape& shape, const char *const* options, const iExpanded2DObstacles* const * preExpanded_Buffer, uint32_t preExpanded_BufferEntries) = 0;
    // Releases unobstructed space previously generated by generateUnobstructedSpaceFor(). 
    virtual void releaseUnobstructedSpaceFor(const iShape& shape) = 0;
    // Generates preprocess data relating to connectivity between unobstructed space boundaries, to support connected region queries. 
    virtual void generateConnectedRegionPreprocessFor(const iShape& shape, const char *const* options) = 0;
    // Releases connected region preprocess previously generated by generateConnectedRegionPreprocessFor(). 
    virtual void releaseConnectedRegionPreprocessFor(const iShape& shape) = 0;
    // Generates preprocess data for pathfinding around agent unobstructed space. 
    virtual void generatePathfindPreprocessFor(const iShape& shape, const char *const* options) = 0;
    // Releases pathfind preprocess previously generated by generatePathfindPreprocessFor(). 
    virtual void releasePathfindPreprocessFor(const iShape& shape) = 0;
    // Tells PathEngine that it is ok to discard any data associated with this mesh that is only required for generating unobstructed space or preprocess. 
    virtual void preprocessGenerationCompleted() = 0;
    // A helper function for checking whether unobstructed space is present for a given shape. 
    virtual bool shapeCanCollide(const iShape& shape) const = 0;
    // A helper function for checking whether combined unobstructed space is present for a given shape. 
    virtual bool shapeHasCombinedUnobstructedSpace(const iShape& shape) const = 0;
    // A helper function for checking whether connected region preprocess is present for a given shape. 
    virtual bool shapeHasConnectedRegionPreprocess(const iShape& shape) const = 0;
    // A helper function for checking whether pathfinding queries are supported for a given shape (and collision context). 
    virtual bool shapeCanPathfind(const iShape& shape, const iCollisionContext* context) const = 0;
    // Saves unobstructed space to persistent storage. 
    virtual void saveUnobstructedSpaceFor(const iShape& shape, iOutputStream& os) const = 0;
    // Loads unobstructed space from persistent storage. 
    virtual void loadUnobstructedSpaceFor(const iShape& shape, char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Saves connected region preprocess to persistent storage. 
    virtual void saveConnectedRegionPreprocessFor(const iShape& shape, iOutputStream& os) const = 0;
    // Loads connected region preprocess from persistent storage. 
    virtual void loadConnectedRegionPreprocessFor(const iShape& shape, char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Saves pathfind preprocess to persistent storage. 
    virtual void savePathfindPreprocessFor(const iShape& shape, iOutputStream& os) const = 0;
    // Loads pathfind preprocess from persistent storage. 
    virtual void loadPathfindPreprocessFor(const iShape& shape, char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    virtual iAgent* placeAgent__Raw(const iShape& shape, const cPosition& position) const = 0;
    virtual iCollisionContext* newContext__Raw() const = 0;
    virtual iObstacleSet* newObstacleSet__Raw() const = 0;
    // Tests whether the specified position is obstructed for agents with the given collision shape. 
    virtual bool testPointCollision(const iShape& shape, const iCollisionContext* context, const cPosition& position) const = 0;
    // Tests whether agents with the given collision shape can move along the specified line without being obstructed. 
    virtual bool testLineCollision(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& end) const = 0;
    // Tests whether agents with the given collision shape can move along the specified line without being obstructed. This version does not require the end of the line to be fully specified. 
    virtual bool testLineCollision_XY(const iShape& shape, const iCollisionContext* context, const cPosition& start, int32_t x, int32_t y, int32_t& cell) const = 0;
    virtual bool firstCollision__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, int32_t x, int32_t y, int32_t& cell, cCollidingLine& collidingLine, iAgent*& agent) const = 0;
    // Finds the closest unobstructed position, if one exists within a specified region around a given position. 
    virtual cPosition findClosestUnobstructedPosition(const iShape& shape, const iCollisionContext* context, const cPosition& position, int32_t maximumDistance) const = 0;
    // Finds the closest unobstructed position to a specified target, within a specified query range and within local mesh as defined by a set of supplied root positions, if one exists (with an optional connectivity constraint based on connected regions preprocess). 
    virtual cPosition findClosestUnobstructedPositionEx(const iShape& shape, const iCollisionContext* context, cPosition const * rootPositions_Buffer, uint32_t rootPositions_BufferEntries, const cHorizontalRange& queryRegion, int32_t targetX, int32_t targetY, int32_t const * permittedConnectedRegions_Buffer, uint32_t permittedConnectedRegions_BufferEntries) const = 0;
    virtual void getAllAgentsOverlapped__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& position, iArrayReceiver<iAgent*>& overlappedAgents) const = 0;
    virtual iPath* findShortestPath__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal) const = 0;
    virtual iPath* findShortestPath_WithQueryCallBack__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal, iQueryCallBack* queryCallBack) const = 0;
    virtual iPath* findPathAway__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& awayFrom, int32_t distanceAway) const = 0;
    virtual iPath* findPathAway_WithQueryCallBack__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& awayFrom, int32_t distanceAway, iQueryCallBack* queryCallBack) const = 0;
    virtual iPath* generateCurvedPath__Raw(const iShape& shape, iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const = 0;
    virtual iPath* constructPath__Raw(cPosition const * positions_Buffer, uint32_t positions_BufferEntries, int32_t const * connectionIndices_Buffer, uint32_t connectionIndices_BufferEntries) const = 0;
    virtual iPath* copyPath__Raw(iPath* path) const = 0;
    virtual iPath* reversePath__Raw(iPath* path) const = 0;
    // Saves internal data relating to a path to persistent storage.
    virtual void savePath(iPath* path, iOutputStream& os) const = 0;
    virtual iPath* loadPath__Raw(char const * data_Buffer, uint32_t data_BufferEntries) const = 0;
    // A helper method for drawing lines that hug the surface of the 3D ground mesh. 
    virtual void renderLineOnGround(const cPosition& start, int32_t endX, int32_t endY, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    virtual iObstacleSet* newObstacleSet_WithAttributes__Raw(const char *const* attributes) const = 0;
    // Attempts to find a cPosition that is both within the specified section and also inside the supplied polygon. 
    virtual cPosition positionInSectionInShape(int32_t sectionID, int32_t shapeOriginX, int32_t shapeOriginY, int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) const = 0;
    virtual iPath* generateCurvedPath_WithEndVector__Raw(const iShape& shape, iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t endVectorX, int32_t endVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const = 0;
    virtual iAgent* placeLargeStaticObstacle__Raw(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries, const cPosition& position) const = 0;
    virtual iAgent* placeProjected3DObstruction__Raw(const iFaceVertexMesh* facesToProject, const cPosition& rootFrom, int32_t agentHeight) const = 0;
    // Adds a set of anchors and shapes, represented in the form of an object derived from the Interface iAnchorsAndPinnedShapes interface class, to this mesh. 
    virtual void addAnchorsAndShapes(const iAnchorsAndPinnedShapes* anchorsAndShapes, const char* idPrefix, int32_t numberOfTerrainLayers) = 0;
    // Enables automatic generation of off-mesh connections based on the generation of sample points around the mesh edge and a callback object that determines whether connection is possible for a given pair of sample points. 
    virtual void autoGenerateConnections(int32_t sampleSpacing, int32_t localityConstraint, int32_t horizontalRange, int32_t verticalRange, int32_t dropRange, iConnectionTestCallBack& callBack) = 0;
    // Enables iteration through the set of obstacles burnt into a mesh. 
    virtual int32_t getNumberOfBurntInObstacles() const = 0;
    // Returns the root position for the specified burnt in obstacle, or an explicitly invalid position (position with cell set to -1) in the case of a 2D obstacle. 
    virtual cPosition getBurntInObstacleRoot(int32_t index) const = 0;
    // Returns the number of vertices in the burnt in obstacle at the given index.
    virtual int32_t getBurntInObstacleVertices(int32_t index) const = 0;
    // Returns the coordinates of the specified burnt in obstacle vertex.
    virtual void getBurntInObstacleVertex(int32_t obstacleIndex, int32_t vertexIndex, int32_t& x, int32_t& y) const = 0;
    // Returns the number of distinct regions in connected region preprocess for the specified shape.
    virtual int32_t getNumberOfConnectedRegions(const iShape& shape) const = 0;
    // Returns an index for the connected region at a given position. This can be used to quickly determine the reachability between positions. 
    virtual int32_t getConnectedRegionFor(const iShape& shape, const cPosition& p) const = 0;
    // Returns an index for the connected region for an agent's current position. 
    virtual int32_t getConnectedRegionForAgent(iAgent& agent) const = 0;
    // Removes any named obstacles stored on this mesh. 
    virtual void clearAllNamedObstacles() = 0;
    // Checks whether a position has been marked as blocked through the surface type traverse costs mechanism. 
    virtual bool positionIsBlockedBySurfaceTypeTraverseCosts(const iCollisionContext* context, const cPosition& position) const = 0;
    // Helper method to obtain the mesh federation tile index. 
    virtual int32_t getFederationTileIndex() const = 0;
    // Calculates the movement cost for a given line, as applied during pathfinding. 
    virtual uint32_t calculatePathfindingCostForLine(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& end, bool& blocked) const = 0;
    // Saves the underlying ground for this iMesh. 
    virtual void saveGroundEx(const char* format, bool includeMapping, bool includeTranslationTo2D, bool includePartitioning, iOutputStream& outputStream) const = 0;
    virtual iPath* findShortestPath_MultipleStartPositions__Raw(const iShape& shape, const iCollisionContext* context, cPosition const * startPositions_Buffer, uint32_t startPositions_BufferEntries, const cPosition& goal, iQueryCallBack* queryCallBack) const = 0;
    // Helper function providing access to the geometry of the underlying 3D mesh. 
    virtual int32_t get3DFaceVertexIndexInMesh(int32_t faceIndex, int32_t vertexIndexInFace) const = 0;
    // A helper method for drawing the boundaries for any base obstacles placed on this mesh. 
    virtual void renderBaseObstacles(int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // A helper method for drawing unobstructed space boundary lines. 
    virtual void renderUnobstructedSpaceBoundaries(const iShape& shape, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // A helper method for drawing the (expanded) unobstructed space boundaries for a pathfinding obstacle. 
    virtual void renderAgentExpansion(const iShape& expandShape, const iAgent& agent, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // A helper method for visualising horizontal range boundaries. 
    virtual void renderRangeBounds(const cHorizontalRange& range, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // A helper method for visualising horizontal range boundaries. 
    virtual void renderRangeBounds_Local(const cHorizontalRange& range, const cPosition& root, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // Helper for constructing a cPosition from horizontal coordinates and 3D face index. 
    virtual cPosition positionForPointIn3DFace(int32_t faceIndex, int32_t x, int32_t y) const = 0;
    // Helper method for testing whether horizontal coordinates are inside a 3D face. 
    virtual bool pointIsIn3DFace(int32_t faceIndex, int32_t x, int32_t y) const = 0;
    // A helper method for visual confirmation of boundaries split off for pathfind space optimisation. 
    virtual void renderUnsplitBoundaries(const iShape& shape, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // A helper method for visual confirmation of boundaries split off for pathfind space optimisation. 
    virtual void renderSplitBoundaries(const iShape& shape, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    // Clears all anchors currently stored on this mesh. 
    virtual void clearAllAnchors() = 0;
    // A helper method for drawing unobstructed space boundary lines. 
    virtual void renderObstacleSetUnobstructedSpace(const iObstacleSet& set, const iShape& shape, int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    virtual iAgent* placeLargeStatic2DObstacle__Raw(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) const = 0;
    // Generates (release 6 type) preprocess data for pathfinding around agent unobstructed space. 
    virtual void generatePathfindPreprocess6For(const iShape& shape, const char *const* options) = 0;
    // A helper function for checking whether pathfinding queries are supported (with release 6 preprocess type) for a given shape (and collision context). 
    virtual bool shapeHasPathfind6Preprocess(const iShape& shape) const = 0;
    virtual iPath* findShortestPath6__Raw(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal, iQueryCallBack* queryCallBack) const = 0;
    virtual iPath* findShortestPath6_MultipleStartPositions__Raw(const iShape& shape, const iCollisionContext* context, cPosition const * startPositions_Buffer, uint32_t startPositions_BufferEntries, const cPosition& goal, iQueryCallBack* queryCallBack) const = 0;
    // Helper for querying local origin of mesh, e.g. for mesh federation tiles. 
    virtual void getOrigin(int32_t& x, int32_t& y) const = 0;
    // Background preprocess generation version of iMesh::generateUnobstructedSpaceFor(). 
    virtual void generateBackgroundUnobstructedSpaceFor(const iShape& shape, bool combine, const char *const* options) = 0;
    // Background preprocess generation version of iMesh::generateUnobstructedSpaceFor_WithPreExpanded(). 
    virtual void generateBackgroundUnobstructedSpaceFor_WithPreExpanded(const iShape& shape, const char *const* options, const iExpanded2DObstacles* const * preExpanded_Buffer, uint32_t preExpanded_BufferEntries) = 0;
    // Background preprocess generation version of iMesh::generateConnectedRegionPreprocessFor(). 
    virtual void generateBackgroundConnectedRegionPreprocessFor(const iShape& shape, const char *const* options) = 0;
    // Background preprocess generation version of iMesh::generatePathfindPreprocessFor(). 
    virtual void generateBackgroundPathfindPreprocessFor(const iShape& shape, const char *const* options) = 0;
    // Promotes background preprocess generated for the specified shape to main preprocess, with this then taking effect for any subsequent queries. 
    virtual void promoteBackgroundPreprocessFor(const iShape& shape) = 0;
    // Releases unobstructed space previously generated by generateBackgroundUnobstructedSpaceFor(). 
    virtual void releaseBackgroundUnobstructedSpaceFor(const iShape& shape) = 0;
    // Background preprocess generation version of iMesh::generatePathfindPreprocess6For(). 
    virtual void generateBackgroundPathfindPreprocess6For(const iShape& shape, const char *const* options) = 0;
    // A helper function to access the ground surface object for a mesh. 
    virtual const iGround& refGround() const = 0;
    virtual const iGround* getGround__Raw() const = 0;
    // Saves unobstructed space to persistent storage. 
    virtual void saveBackgroundUnobstructedSpaceFor(const iShape& shape, iOutputStream& os) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iMesh> addExternalRef()
    {
        auto result = std::unique_ptr<iMesh>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iMesh> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iMesh>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iMesh* afterCast = static_cast<iMesh*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Construct an agent from obstacle information stored with the mesh. 
    std::unique_ptr<iAgent> retrieveNamedObstacle(const char* id) const 
    {
        auto result = std::unique_ptr<iAgent>(retrieveNamedObstacle__Raw(id));
        return result;
    }
    // Used for iterating through fixed obstacles. 
    void retrieveNamedObstacleByIndex(int32_t index, std::unique_ptr<iAgent>& agent, const char*& id) const 
    {
        iAgent* agent__Raw;
        retrieveNamedObstacleByIndex__Raw(index, agent__Raw, id);
        agent = std::unique_ptr<iAgent>(agent__Raw);
    }
    // Constructs an Interface iAgent object to represent placement of an object of a given shape at a point on the surface of this mesh. 
    std::unique_ptr<iAgent> placeAgent(const iShape& shape, const cPosition& position) const 
    {
        auto result = std::unique_ptr<iAgent>(placeAgent__Raw(shape, position));
        return result;
    }
    // Creates an Interface iCollisionContext to representing a state of dynamic collision for this mesh. 
    std::unique_ptr<iCollisionContext> newContext() const 
    {
        auto result = std::unique_ptr<iCollisionContext>(newContext__Raw());
        return result;
    }
    // Creates a free standing obstacle set. 
    std::unique_ptr<iObstacleSet> newObstacleSet() const 
    {
        auto result = std::unique_ptr<iObstacleSet>(newObstacleSet__Raw());
        return result;
    }
    // Determines the first contact, if any, for an agent with the specified collision shape moving along the specified line. 
    bool firstCollision(const iShape& shape, const iCollisionContext* context, const cPosition& start, int32_t x, int32_t y, int32_t& cell, cCollidingLine& collidingLine, std::unique_ptr<iAgent>& agent) const 
    {
        iAgent* agent__Raw;
        auto result = firstCollision__Raw(shape, context, start, x, y, cell, collidingLine, agent__Raw);
        agent = std::unique_ptr<iAgent>(agent__Raw);
        return result;
    }
    // Fills a buffer with the set of agents in a given context overlapped by the given shape at the given position. 
    template <class tVector> void getAllAgentsOverlapped(const iShape& shape, const iCollisionContext* context, const cPosition& position, tVector& overlappedAgents) const 
    {
        cAPIRefReceiver<tVector> overlappedAgents__Receiver(overlappedAgents);
        getAllAgentsOverlapped__Raw(shape, context, position, overlappedAgents__Receiver);
    }
    // Finds the shortest path from a given start position to a given end position for agents with a specified collision shape. 
    std::unique_ptr<iPath> findShortestPath(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPath__Raw(shape, context, start, goal));
        return result;
    }
    // Finds the shortest path from a given start position to a given end position for agents with a specified collision shape.
    // (This version supports passing in a callback object to enable the query to be aborted early.) 
    std::unique_ptr<iPath> findShortestPath_WithQueryCallBack(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPath_WithQueryCallBack__Raw(shape, context, start, goal, queryCallBack));
        return result;
    }
    // Finds the shortest path of escape from a specified region from a given start position. 
    std::unique_ptr<iPath> findPathAway(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& awayFrom, int32_t distanceAway) const 
    {
        auto result = std::unique_ptr<iPath>(findPathAway__Raw(shape, context, start, awayFrom, distanceAway));
        return result;
    }
    // Finds the shortest path of escape from a specified region from a given start position.
    // (This version supports passing in a callback object to enable the query to be aborted early.) 
    std::unique_ptr<iPath> findPathAway_WithQueryCallBack(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& awayFrom, int32_t distanceAway, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findPathAway_WithQueryCallBack__Raw(shape, context, start, awayFrom, distanceAway, queryCallBack));
        return result;
    }
    // Post process path smoothing.
    // Returns a new path based on a 'base path', but with corners smoothed according to supplied parameters. 
    std::unique_ptr<iPath> generateCurvedPath(const iShape& shape, iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const 
    {
        auto result = std::unique_ptr<iPath>(generateCurvedPath__Raw(shape, basePath, context, startVectorX, startVectorY, sectionLength, turnRatio1, turnRatio2));
        return result;
    }
    // Provided to enable Interface iPath objects to be constructed directly from a sequence of positions. 
    std::unique_ptr<iPath> constructPath(cPosition const * positions_Buffer, uint32_t positions_BufferEntries, int32_t const * connectionIndices_Buffer, uint32_t connectionIndices_BufferEntries) const 
    {
        auto result = std::unique_ptr<iPath>(constructPath__Raw(positions_Buffer, positions_BufferEntries, connectionIndices_Buffer, connectionIndices_BufferEntries));
        return result;
    }
    // Creates a copy of a path object. 
    std::unique_ptr<iPath> copyPath(iPath* path) const 
    {
        auto result = std::unique_ptr<iPath>(copyPath__Raw(path));
        return result;
    }
    // Creates a reversed copy of a path. 
    std::unique_ptr<iPath> reversePath(iPath* path) const 
    {
        auto result = std::unique_ptr<iPath>(reversePath__Raw(path));
        return result;
    }
    // Loads internal data relating to a path from persistent storage.
    std::unique_ptr<iPath> loadPath(char const * data_Buffer, uint32_t data_BufferEntries) const 
    {
        auto result = std::unique_ptr<iPath>(loadPath__Raw(data_Buffer, data_BufferEntries));
        return result;
    }
    // Creates a free standing obstacle set. 
    std::unique_ptr<iObstacleSet> newObstacleSet_WithAttributes(const char *const* attributes) const 
    {
        auto result = std::unique_ptr<iObstacleSet>(newObstacleSet_WithAttributes__Raw(attributes));
        return result;
    }
    // Post process path smoothing.
    // Returns a new path based on a 'base path', but with corners smoothed according to supplied parameters. 
    std::unique_ptr<iPath> generateCurvedPath_WithEndVector(const iShape& shape, iPath* basePath, const iCollisionContext* context, int32_t startVectorX, int32_t startVectorY, int32_t endVectorX, int32_t endVectorY, int32_t sectionLength, float turnRatio1, float turnRatio2) const 
    {
        auto result = std::unique_ptr<iPath>(generateCurvedPath_WithEndVector__Raw(shape, basePath, context, startVectorX, startVectorY, endVectorX, endVectorY, sectionLength, turnRatio1, turnRatio2));
        return result;
    }
    // Constructs an Interface iAgent object to represent placement of an obstacle on this mesh.
    // This version enables shapes to be placed without the application of the usual range constraint on Interface iShape. 
    std::unique_ptr<iAgent> placeLargeStaticObstacle(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries, const cPosition& position) const 
    {
        auto result = std::unique_ptr<iAgent>(placeLargeStaticObstacle__Raw(shapeCoordinates_Buffer, shapeCoordinates_BufferEntries, position));
        return result;
    }
    // Constructs an Interface iAgent object to represent the way in which a piece of 3D geometry obstructs a specified ground layer. 
    std::unique_ptr<iAgent> placeProjected3DObstruction(const iFaceVertexMesh* facesToProject, const cPosition& rootFrom, int32_t agentHeight) const 
    {
        auto result = std::unique_ptr<iAgent>(placeProjected3DObstruction__Raw(facesToProject, rootFrom, agentHeight));
        return result;
    }
    // Finds the shortest path from a given set of start positions to a given end position for agents with a specified collision shape.
    //  
    std::unique_ptr<iPath> findShortestPath_MultipleStartPositions(const iShape& shape, const iCollisionContext* context, cPosition const * startPositions_Buffer, uint32_t startPositions_BufferEntries, const cPosition& goal, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPath_MultipleStartPositions__Raw(shape, context, startPositions_Buffer, startPositions_BufferEntries, goal, queryCallBack));
        return result;
    }
    // Constructs an Interface iAgent object to represent placement of an obstacle on this mesh.
    // This version doesn't require a root position, with the obstacle then applying to all ground layers overlapping the obstacle horizontally. 
    std::unique_ptr<iAgent> placeLargeStatic2DObstacle(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) const 
    {
        auto result = std::unique_ptr<iAgent>(placeLargeStatic2DObstacle__Raw(shapeCoordinates_Buffer, shapeCoordinates_BufferEntries));
        return result;
    }
    // Finds the shortest path from a given start position to a given end position for agents with a specified collision shape. 
    std::unique_ptr<iPath> findShortestPath6(const iShape& shape, const iCollisionContext* context, const cPosition& start, const cPosition& goal, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPath6__Raw(shape, context, start, goal, queryCallBack));
        return result;
    }
    // Finds the shortest path from a given set of start positions to a given end position for agents with a specified collision shape. 
    std::unique_ptr<iPath> findShortestPath6_MultipleStartPositions(const iShape& shape, const iCollisionContext* context, cPosition const * startPositions_Buffer, uint32_t startPositions_BufferEntries, const cPosition& goal, iQueryCallBack* queryCallBack) const 
    {
        auto result = std::unique_ptr<iPath>(findShortestPath6_MultipleStartPositions__Raw(shape, context, startPositions_Buffer, startPositions_BufferEntries, goal, queryCallBack));
        return result;
    }
    // A helper function to access the ground surface object for a mesh. 
    std::unique_ptr<const iGround> getGround() const 
    {
        auto result = std::unique_ptr<const iGround>(getGround__Raw());
        return result;
    }
};
#endif

#ifndef PATHENGINE_iPathEngine_DEFINED
#define PATHENGINE_iPathEngine_DEFINED
class iPathEngine
{
public:
    // Obtains a set of attributes for the object that provides this interface.
    virtual const char *const* getVersionAttributes() const = 0;
    // Returns the major version number for the code supporting this interface. 
    virtual int32_t getInterfaceMajorVersion() const = 0;
    // Returns the minor version number for the code supporting this interface. 
    virtual int32_t getInterfaceMinorVersion() const = 0;
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iPathEngine* addExternalRef__Raw() = 0;
    virtual const iPathEngine* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Returns release numbers for the code supporting this interface.
    virtual void getReleaseNumbers(int32_t& majorReleaseNumber, int32_t& minorReleaseNumber, int32_t& internalReleaseNumber) const = 0;
    // Controls PathEngine's path caching mechanism. 
    virtual void setPathPoolParameters(int32_t pathsPooled, int32_t pointsPerPath) = 0;
    // Checks whether coordinates are valid for iPathEngine::newShape(). 
    virtual bool shapeIsValid(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) = 0;
    virtual iShape* newShape__Raw(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) = 0;
    virtual iMesh* loadMeshFromBuffer__Raw(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, const char *const* options) = 0;
    virtual iMesh* buildMeshFromContent__Raw(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options) = 0;
    // (Depreciated in favour of iPathEngine::save2DContentSnapshot())
    // Provided for generating snapshots of the source content when reporting bugs in the content processing functionality. 
    virtual void saveContentData(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char* format, iOutputStream& os) = 0;
    // Seeds PathEngine's internal pseudo-random sequence.
    virtual void setRandomSeed(uint32_t value) = 0;
    // Used to check if persistent data saved out for unobstructed space from an older version of the library can be loaded into this version. 
    virtual bool unobstructedSpaceVersionIsCompatible(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Used to check if persistent data saved out for connected region preprocess from an older version of the library can be loaded into this version. 
    virtual bool connectedRegionPreprocessVersionIsCompatible(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Used to check if persistent data saved out for pathfind preprocess from an older version of the library can be loaded into this version. 
    virtual bool pathfindPreprocessVersionIsCompatible(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    virtual iContentChunk* newContentChunk__Raw(const iFaceVertexMesh* ground, const iAnchorsAndPinnedShapes* anchorsAndShapes, const char *const* attributes) = 0;
    virtual iContentChunk* loadContentChunk__Raw(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    virtual iMeshFederation* buildMeshFederation_FromWorldMesh__Raw(const iMesh& worldMesh, int32_t tileStep, int32_t overlap, iMeshFederationTileGeneratedCallBack& tileGeneratedCallBack) = 0;
    virtual iMeshFederation* loadFederation__Raw(const char* format, char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    virtual iMeshFederation* buildMeshFederation_TilingOnly__Raw(const cHorizontalRange& worldRange, int32_t tileStep, int32_t overlap) = 0;
    // Checks whether coordinates are valid for iMesh::placeLargeStaticObstacle(). 
    virtual bool largeStaticObstacleShapeIsValid(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries, int32_t x, int32_t y) = 0;
    virtual iSourceTiling* buildRegularSourceTiling__Raw(const cHorizontalRange& worldRange, int32_t tileSize) = 0;
    virtual iMesh* buildMeshFromGroundTiles__Raw(const iSourceTiling& tiling, const iMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options) = 0;
    virtual iSourceTiling* loadSourceTiling__Raw(const char* format, char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Provided for generating snapshots of the source content when reporting bugs in the content processing functionality. 
    virtual void save2DContentSnapshot(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options, const char* format, iOutputStream& os) = 0;
    virtual iSourceTiling* buildRegularSourceTiling_RoundUpForVoxels__Raw(const cHorizontalRange& worldRange, int32_t tileSize, int32_t voxelSize) = 0;
    virtual iMesh* buildMeshByCopyingConnectedGround__Raw(const iMesh& baseMesh, cPosition const * rootPositions_Buffer, uint32_t rootPositions_BufferEntries, const char *const* options) = 0;
    virtual iMesh* buildMeshAroundTiledObstructions__Raw(char const * data_Buffer, uint32_t data_BufferEntries, int32_t tilesInX, int32_t tilesInY, int32_t originX, int32_t originY, int32_t tileSize, const char *const* options) = 0;
    virtual iMesh* buildUnobstructedSpaceGround__Raw(const iMesh& mesh, const iShape& shape, const char *const* options) = 0;
    // Checks whether coordinates are valid for iMesh::placeLargeStatic2DObstacle(). 
    virtual bool largeStatic2DObstacleShapeIsValid(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries) = 0;
    virtual iMesh* buildMeshAroundObstructionBoundaries__Raw(int32_t const * boundaries_Buffer, uint32_t boundaries_BufferEntries, bool optimise, int32_t optimisationThreshold_Numerator, int32_t optimisationThreshold_Denominator, const char *const* options) = 0;
    virtual iExpanded2DObstacles* preExpandObstacles__Raw(int32_t originX, int32_t originY, int32_t const * obstacles_Buffer, uint32_t obstacles_BufferEntries, const iShape& shape, bool optimise, int32_t optimiseHorizontalRange) = 0;
    virtual iExpanded2DObstacles* loadExpanded2DObstacles__Raw(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    virtual iShape* newShape_FromExpanded2DObstacles__Raw(const iExpanded2DObstacles& preExpanded) = 0;
    virtual iMesh* buildExpanded2DObstaclesContainerMesh__Raw(const iExpanded2DObstacles& preExpanded, iShape& expandShape) = 0;
    // A helper method for saving face/vertex mesh data to persistence. 
    virtual void saveFaceVertexMesh(const iFaceVertexMesh& mesh, const char* format, iOutputStream& os) = 0;
    // A helper method for restoring face/vertex mesh data from persistence. 
    virtual void loadFaceVertexMesh(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, iFaceVertexBuilder& faceVertexBuilder) = 0;
    // Extracts a set of convex polygons that can be used as a 2D collision footprint for arbitrary 3D mesh data. 
    virtual void extractConvexPolyFootprint(const iFaceVertexMesh& mesh, int32_t optimisationHorizontalRange, iArrayReceiver<int32_t>& result) = 0;
    virtual iMesh* newMesh__Raw(const iGround& ground, const char *const* options) = 0;
    // A helper function to access the root of PathEngine's directed acyclic ownership graph. 
    virtual const iOwnershipGraphRoot& refOwnershipGraphRoot() const = 0;
    virtual const iOwnershipGraphRoot* getOwnershipGraphRoot__Raw() const = 0;
    virtual const iGround* extractGroundFootprint__Raw(const iFaceVertexMesh& mesh, int32_t optimisationHorizontalRange, const char *const* options) = 0;
    virtual const iGround* loadGroundFromBuffer__Raw(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, const char *const* options) = 0;
    virtual iMesh* buildMeshFromGroundTiles_IGround__Raw(const iSourceTiling& tiling, const iGround* const * groundTiles_Buffer, uint32_t groundTiles_BufferEntries, const char *const* options) = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iPathEngine> addExternalRef()
    {
        auto result = std::unique_ptr<iPathEngine>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iPathEngine> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iPathEngine>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iPathEngine* afterCast = static_cast<iPathEngine*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Creates a new iShape object. 
    std::unique_ptr<iShape> newShape(int32_t const * shapeCoordinates_Buffer, uint32_t shapeCoordinates_BufferEntries)
    {
        auto result = std::unique_ptr<iShape>(newShape__Raw(shapeCoordinates_Buffer, shapeCoordinates_BufferEntries));
        return result;
    }
    // Constructs an iMesh object from a data buffer containing a 3d ground mesh. 
    std::unique_ptr<iMesh> loadMeshFromBuffer(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(loadMeshFromBuffer__Raw(format, data_Buffer, data_BufferEntries, options));
        return result;
    }
    // Constructs an iMesh object by processesing geometry specified in 'face/vertex' form. 
    std::unique_ptr<iMesh> buildMeshFromContent(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshFromContent__Raw(meshes_Buffer, meshes_BufferEntries, options));
        return result;
    }
    // Constructs a content chunk object which can then be saved out and later loaded for instantiation in a separate content tool. 
    std::unique_ptr<iContentChunk> newContentChunk(const iFaceVertexMesh* ground, const iAnchorsAndPinnedShapes* anchorsAndShapes, const char *const* attributes)
    {
        auto result = std::unique_ptr<iContentChunk>(newContentChunk__Raw(ground, anchorsAndShapes, attributes));
        return result;
    }
    // Constructs an iContentChunk object from a data buffer previously saved out by iContentChunk::save(). 
    std::unique_ptr<iContentChunk> loadContentChunk(char const * data_Buffer, uint32_t data_BufferEntries)
    {
        auto result = std::unique_ptr<iContentChunk>(loadContentChunk__Raw(data_Buffer, data_BufferEntries));
        return result;
    }
    // Returns a newly created Interface iMeshFederation object, and generates the corresponding set of federated Interface iMeshes, from a single large 'world mesh', with range determined by the range of worldMesh, and with the specified tile size, and overlap. 
    std::unique_ptr<iMeshFederation> buildMeshFederation_FromWorldMesh(const iMesh& worldMesh, int32_t tileStep, int32_t overlap, iMeshFederationTileGeneratedCallBack& tileGeneratedCallBack)
    {
        auto result = std::unique_ptr<iMeshFederation>(buildMeshFederation_FromWorldMesh__Raw(worldMesh, tileStep, overlap, tileGeneratedCallBack));
        return result;
    }
    // Constructs an iMeshFederation object from a data buffer previously saved out by iMeshFederation::save(). 
    std::unique_ptr<iMeshFederation> loadFederation(const char* format, char const * data_Buffer, uint32_t data_BufferEntries)
    {
        auto result = std::unique_ptr<iMeshFederation>(loadFederation__Raw(format, data_Buffer, data_BufferEntries));
        return result;
    }
    // Returns a newly created Interface iMeshFederation object, to encapsulate a mesh federation tiling, but without actually generating any federated Interface iMesh instances. 
    std::unique_ptr<iMeshFederation> buildMeshFederation_TilingOnly(const cHorizontalRange& worldRange, int32_t tileStep, int32_t overlap)
    {
        auto result = std::unique_ptr<iMeshFederation>(buildMeshFederation_TilingOnly__Raw(worldRange, tileStep, overlap));
        return result;
    }
    // Returns a newly created Interface iSourceTiling object that can be used to perform 'tile-by-tile' content processing. 
    std::unique_ptr<iSourceTiling> buildRegularSourceTiling(const cHorizontalRange& worldRange, int32_t tileSize)
    {
        auto result = std::unique_ptr<iSourceTiling>(buildRegularSourceTiling__Raw(worldRange, tileSize));
        return result;
    }
    // Constructs a single iMesh object by connecting multiple tiled ground pieces together at tile borders. 
    std::unique_ptr<iMesh> buildMeshFromGroundTiles(const iSourceTiling& tiling, const iMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshFromGroundTiles__Raw(tiling, meshes_Buffer, meshes_BufferEntries, options));
        return result;
    }
    // Constructs an iSourceTiling object from a data buffer previously saved out by iSourceTiling::save(). 
    std::unique_ptr<iSourceTiling> loadSourceTiling(const char* format, char const * data_Buffer, uint32_t data_BufferEntries)
    {
        auto result = std::unique_ptr<iSourceTiling>(loadSourceTiling__Raw(format, data_Buffer, data_BufferEntries));
        return result;
    }
    // Returns a newly created Interface iSourceTiling object that can be used to perform 'tile-by-tile' content processing. 
    std::unique_ptr<iSourceTiling> buildRegularSourceTiling_RoundUpForVoxels(const cHorizontalRange& worldRange, int32_t tileSize, int32_t voxelSize)
    {
        auto result = std::unique_ptr<iSourceTiling>(buildRegularSourceTiling_RoundUpForVoxels__Raw(worldRange, tileSize, voxelSize));
        return result;
    }
    // A helper method for stripping out unconnected ground mesh pieces, based on a supplied set of 'root positions'. 
    std::unique_ptr<iMesh> buildMeshByCopyingConnectedGround(const iMesh& baseMesh, cPosition const * rootPositions_Buffer, uint32_t rootPositions_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshByCopyingConnectedGround__Raw(baseMesh, rootPositions_Buffer, rootPositions_BufferEntries, options));
        return result;
    }
    // A helper method for building ground mesh around a set of tiled obstructions. 
    std::unique_ptr<iMesh> buildMeshAroundTiledObstructions(char const * data_Buffer, uint32_t data_BufferEntries, int32_t tilesInX, int32_t tilesInY, int32_t originX, int32_t originY, int32_t tileSize, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshAroundTiledObstructions__Raw(data_Buffer, data_BufferEntries, tilesInX, tilesInY, originX, originY, tileSize, options));
        return result;
    }
    // A helper method for generating a ground mesh representation of agent unobstructed space. 
    std::unique_ptr<iMesh> buildUnobstructedSpaceGround(const iMesh& mesh, const iShape& shape, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildUnobstructedSpaceGround__Raw(mesh, shape, options));
        return result;
    }
    // A helper method for building ground mesh around a set of 2D obstruction boundaries. 
    std::unique_ptr<iMesh> buildMeshAroundObstructionBoundaries(int32_t const * boundaries_Buffer, uint32_t boundaries_BufferEntries, bool optimise, int32_t optimisationThreshold_Numerator, int32_t optimisationThreshold_Denominator, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshAroundObstructionBoundaries__Raw(boundaries_Buffer, boundaries_BufferEntries, optimise, optimisationThreshold_Numerator, optimisationThreshold_Denominator, options));
        return result;
    }
    // Pre-expands and combines a set of convex 2D obstacles, with optional boundary optimisation. 
    std::unique_ptr<iExpanded2DObstacles> preExpandObstacles(int32_t originX, int32_t originY, int32_t const * obstacles_Buffer, uint32_t obstacles_BufferEntries, const iShape& shape, bool optimise, int32_t optimiseHorizontalRange)
    {
        auto result = std::unique_ptr<iExpanded2DObstacles>(preExpandObstacles__Raw(originX, originY, obstacles_Buffer, obstacles_BufferEntries, shape, optimise, optimiseHorizontalRange));
        return result;
    }
    // Constructs an iExpanded2DObstacles object from a data buffer previously saved out by iExpanded2DObstacles::save(). 
    std::unique_ptr<iExpanded2DObstacles> loadExpanded2DObstacles(char const * data_Buffer, uint32_t data_BufferEntries)
    {
        auto result = std::unique_ptr<iExpanded2DObstacles>(loadExpanded2DObstacles__Raw(data_Buffer, data_BufferEntries));
        return result;
    }
    // Helper method for setting up an iShape with expansion coordinates from and expanded 2D obstacles instance. 
    std::unique_ptr<iShape> newShape_FromExpanded2DObstacles(const iExpanded2DObstacles& preExpanded)
    {
        auto result = std::unique_ptr<iShape>(newShape_FromExpanded2DObstacles__Raw(preExpanded));
        return result;
    }
    // Helper method for building an iMesh instance that can be used for the purposes of visualising the boundaries of iExpanded2DObstacles instances. 
    std::unique_ptr<iMesh> buildExpanded2DObstaclesContainerMesh(const iExpanded2DObstacles& preExpanded, iShape& expandShape)
    {
        auto result = std::unique_ptr<iMesh>(buildExpanded2DObstaclesContainerMesh__Raw(preExpanded, expandShape));
        return result;
    }
    // Constructs an iMesh object for a given ground surface. 
    std::unique_ptr<iMesh> newMesh(const iGround& ground, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(newMesh__Raw(ground, options));
        return result;
    }
    // A helper function to access the root of PathEngine's directed acyclic ownership graph. 
    std::unique_ptr<const iOwnershipGraphRoot> getOwnershipGraphRoot() const 
    {
        auto result = std::unique_ptr<const iOwnershipGraphRoot>(getOwnershipGraphRoot__Raw());
        return result;
    }
    // Extracts a ground mesh corresponding to the horizontal shape of arbitrary 3D mesh data. 
    std::unique_ptr<const iGround> extractGroundFootprint(const iFaceVertexMesh& mesh, int32_t optimisationHorizontalRange, const char *const* options)
    {
        auto result = std::unique_ptr<const iGround>(extractGroundFootprint__Raw(mesh, optimisationHorizontalRange, options));
        return result;
    }
    // Constructs an iGround object from a data buffer containing a 3d ground mesh. 
    std::unique_ptr<const iGround> loadGroundFromBuffer(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<const iGround>(loadGroundFromBuffer__Raw(format, data_Buffer, data_BufferEntries, options));
        return result;
    }
    // Constructs a single iMesh object by connecting multiple tiled ground pieces together at tile borders. 
    std::unique_ptr<iMesh> buildMeshFromGroundTiles_IGround(const iSourceTiling& tiling, const iGround* const * groundTiles_Buffer, uint32_t groundTiles_BufferEntries, const char *const* options)
    {
        auto result = std::unique_ptr<iMesh>(buildMeshFromGroundTiles_IGround__Raw(tiling, groundTiles_Buffer, groundTiles_BufferEntries, options));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iShape_DEFINED
#define PATHENGINE_iShape_DEFINED
class iShape
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iShape* addExternalRef__Raw() = 0;
    virtual const iShape* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Helper function to return the number of vertices in the shape. 
    virtual int32_t size() const = 0;
    // Helper function for obtaining coordinates of vertices in the shape. 
    virtual void vertex(int32_t index, int32_t& x, int32_t& y) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iShape> addExternalRef()
    {
        auto result = std::unique_ptr<iShape>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iShape> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iShape>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iShape* afterCast = static_cast<iShape*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

#ifndef PATHENGINE_iCollisionContext_DEFINED
#define PATHENGINE_iCollisionContext_DEFINED
class iCollisionContext
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iCollisionContext* addExternalRef__Raw() = 0;
    virtual const iCollisionContext* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Adds a specified agent to this context.
    virtual void addAgent(iAgent& agent) = 0;
    // Removes a specified agent from this context.
    virtual void removeAgent(iAgent& agent) = 0;
    // Tells you how many agents are currently included in the context.
    virtual int32_t getNumberOfAgents() const = 0;
    virtual iAgent* getAgent__Raw(int32_t index) const = 0;
    // Tells you if a specified agent is included directly in this context. 
    virtual bool includes(iAgent& agent) const = 0;
    // Adds a free standing obstacle set to this context. 
    virtual void addObstacleSet(iObstacleSet& context) = 0;
    // Removes a free standing obstacle set to this context.
    virtual void removeObstacleSet(iObstacleSet& context) = 0;
    // Helper method for iterating through obstacle sets in a context.
    virtual int32_t getNumberOfObstacleSets() const = 0;
    virtual iObstacleSet* getObstacleSet__Raw(int32_t index) const = 0;
    // Enables the application to force generation of pathfinding preprocess associated with the collision context.
    virtual void updatePathfindingPreprocessFor(const iShape& shape) const = 0;
    // Associates a traverse cost with regions marked with a specified surface type. 
    virtual void setSurfaceTypeTraverseCost(int32_t surfaceType, float cost) = 0;
    // Associates a direction with the traverse cost for regions marked with a specified surface type. 
    virtual void setSurfaceTypeTraverseCostDirection(int32_t surfaceType, int32_t directionVectorX, int32_t directionVectorY) = 0;
    // Sets a rectangular region outside which PathEngine may discard potential paths. 
    virtual void setQueryBounds(const cHorizontalRange& bounds) = 0;
    // Clears any query bounds previously set for this context. 
    virtual void clearQueryBounds() = 0;
    // Enables the penalty values associated with off-mesh connection to be overlaid and managed dynamically. 
    virtual void setOverlayConnectionPenalty(int32_t connectionIndex, int32_t penalty) = 0;
    // Clears any connection penalties previously set with iCollisionContext::setOverlayConnectionPenalty(). 
    virtual void clearOverlayConnectionPenalties() = 0;
    // Helper method for testing if any overlay connection penalties are set for a context. 
    virtual bool hasOverlayConnectionPenalties() const = 0;
    // Helper method for querying the value of overlay connection penalties set previously for this context (with iCollisionContext::setOverlayConnectionPenalty()). 
    virtual int32_t getOverlayConnectionPenalty(int32_t connectionIndex) const = 0;
    // Tells PathEngine about the possibility for agents to move 'off-mesh' between a pair of endpoints, with this movement possibility specified completely dynamically at run-time. 
    virtual int32_t addRunTimeOffMeshConnection(const cPosition& fromPosition, const cPosition& toPosition, int32_t forwardPenalty, int32_t backPenalty, int32_t forwardConnectionID, int32_t backConnectionID) = 0;
    // Removes a run-time off-mesh connection previously added to this context with iCollisionContext::addRunTimeOffMeshConnection(). 
    virtual void removeRunTimeOffMeshConnection(int32_t connectionHandle) = 0;
    // Removes any run-time off-mesh connections associated with this context. 
    virtual void clearRunTimeOffMeshConnections() = 0;
    // Flags a specified agent as excluded from the context collision state, without invalidating cached state data.
    virtual void temporarilyIgnoreAgent(iAgent& agent) = 0;
    // Restores an agent that was previously marked as excluded from collision state.
    virtual void restoreTemporarilyIgnoredAgent(iAgent& agent) = 0;
    // Returns the number of connected regions in connected region preprocess associated with this object.
    virtual int32_t getNumberOfConnectedRegions(const iShape& shape) const = 0;
    // Returns an index for the connected region at a given position. This can be used to quickly determine the reachability between positions. 
    virtual int32_t getConnectedRegionFor(const iShape& shape, const cPosition& p) const = 0;
    // Returns an index for the connected region for an agent's current position. 
    virtual int32_t getConnectedRegionForAgent(iAgent& agent) const = 0;
    // Removes all agents. 
    virtual void clear() = 0;
    // Provided for iterating through the set of agents in a context.
    virtual iAgent& refAgent(int32_t index) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iCollisionContext> addExternalRef()
    {
        auto result = std::unique_ptr<iCollisionContext>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iCollisionContext> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iCollisionContext>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iCollisionContext* afterCast = static_cast<iCollisionContext*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Provided for iterating through the set of agents in a context.
    std::unique_ptr<iAgent> getAgent(int32_t index) const 
    {
        auto result = std::unique_ptr<iAgent>(getAgent__Raw(index));
        return result;
    }
    // Helper method for iterating through obstacle sets in a context.
    std::unique_ptr<iObstacleSet> getObstacleSet(int32_t index) const 
    {
        auto result = std::unique_ptr<iObstacleSet>(getObstacleSet__Raw(index));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iObstacleSet_DEFINED
#define PATHENGINE_iObstacleSet_DEFINED
class iObstacleSet
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iObstacleSet* addExternalRef__Raw() = 0;
    virtual const iObstacleSet* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Adds a specified agent to this obstacle set.
    virtual void addAgent(iAgent& agent) = 0;
    // Removes a specified agent from this set.
    virtual void removeAgent(iAgent& agent) = 0;
    // Tells you how many agents are currently included in the set.
    virtual int32_t getNumberOfAgents() const = 0;
    virtual iAgent* getAgent__Raw(int32_t index) const = 0;
    // Tells you if a specified agent is included in this set. 
    virtual bool includes(iAgent& agent) const = 0;
    // Equivalent to calling getNumberOfAgents()
    virtual int32_t size() const = 0;
    // Returns true if the set is empty.
    virtual bool empty() const = 0;
    // Checks the validity of preprocess, on obstacle sets marked for preprocessing.
    virtual bool pathfindPreprocessNeedsUpdate(const iShape& shape) const = 0;
    // Enables the application to force generation of pathfind preprocess associated with the obstacle set, for preprocessed obstacle sets.
    virtual void updatePathfindPreprocessFor(const iShape& shape) const = 0;
    // Saves  preprocess to persistent storage. 
    virtual void savePathfindPreprocessFor(const iShape& shape, iOutputStream& os) const = 0;
    // Loads preprocess from persistent storage. 
    virtual void loadPathfindPreprocessFor(const iShape& shape, char const * data_Buffer, uint32_t data_BufferEntries) const = 0;
    // Checks the validity of preprocess, on obstacle sets marked for preprocessing.
    virtual bool connectedRegionPreprocessNeedsUpdate(const iShape& shape) const = 0;
    // Enables the application to force generation of pathfind preprocess associated with the obstacle set, for preprocessed obstacle sets.
    virtual void updateConnectedRegionPreprocessFor(const iShape& shape) const = 0;
    // Returns the number of distinct regions in connected region preprocess for the specified shape.
    virtual int32_t getNumberOfConnectedRegions(const iShape& shape) const = 0;
    // Returns an index for the connected region at a given position. This can be used to quickly determine the reachability between positions. 
    virtual int32_t getConnectedRegionFor(const iShape& shape, const cPosition& p) const = 0;
    // Returns an index for the connected region for an agent's current position. 
    virtual int32_t getConnectedRegionForAgent(iAgent& agent) const = 0;
    // Removes all agents. 
    virtual void clear() = 0;
    // Flags a specified agent as excluded from the obstacle set collision state, without invalidating cached state data.
    virtual void temporarilyIgnoreAgent(iAgent& agent) = 0;
    // Restores an agent that was previously marked as excluded from collision state.
    virtual void restoreTemporarilyIgnoredAgent(iAgent& agent) = 0;
    // Provided for iterating through the agents in an obstacle set.
    virtual iAgent& refAgent(int32_t index) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iObstacleSet> addExternalRef()
    {
        auto result = std::unique_ptr<iObstacleSet>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iObstacleSet> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iObstacleSet>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iObstacleSet* afterCast = static_cast<iObstacleSet*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Provided for iterating through the agents in the set.
    std::unique_ptr<iAgent> getAgent(int32_t index) const 
    {
        auto result = std::unique_ptr<iAgent>(getAgent__Raw(index));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iPath_DEFINED
#define PATHENGINE_iPath_DEFINED
class iPath
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iPath* addExternalRef__Raw() = 0;
    virtual const iPath* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Returns the number of points that comprise the path. 
    virtual int32_t size() const = 0;
    // Returns a position in the path at a specified index. 
    virtual cPosition position(int32_t pointIndex) const = 0;
    // Enables the client application to determine whether a given segment in a path corresponds to an 'off-mesh connection'. 
    virtual int32_t connectionIndex(int32_t segmentIndex) const = 0;
    // Returns the total length of the path. 
    virtual uint32_t getLength() const = 0;
    // Provides the same functionality as iAgent::advanceAlongPathWithPrecision(), but without requiring creation of an iAgent. 
    virtual bool advanceAlong(const iShape& shape, float distance, const iCollisionContext* context, float& precisionX, float& precisionY) = 0;
    virtual bool advanceAlongWithCollisionInfo__Raw(const iShape& shape, float distance, const iCollisionContext* context, float& precisionX, float& precisionY, cCollidingLine& collidingLine, iAgent*& agent) = 0;
    // A helper method for drawing paths flush with the surface of the 3D ground mesh. 
    virtual void renderOnGround(int32_t originX, int32_t originY, iRender3DLinesCallBack& callBack) const = 0;
    virtual iMesh* getMesh__Raw() const = 0;
    // A helper function to access the mesh on which this path was created. 
    virtual iMesh& refMesh() const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iPath> addExternalRef()
    {
        auto result = std::unique_ptr<iPath>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iPath> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iPath>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iPath* afterCast = static_cast<iPath*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Provides the same functionality as iAgent::advanceAlongPathWithCollisionInfo(), but without requiring creation of an iAgent. 
    bool advanceAlongWithCollisionInfo(const iShape& shape, float distance, const iCollisionContext* context, float& precisionX, float& precisionY, cCollidingLine& collidingLine, std::unique_ptr<iAgent>& agent)
    {
        iAgent* agent__Raw;
        auto result = advanceAlongWithCollisionInfo__Raw(shape, distance, context, precisionX, precisionY, collidingLine, agent__Raw);
        agent = std::unique_ptr<iAgent>(agent__Raw);
        return result;
    }
    // A helper function to access the mesh on which this path was created. 
    std::unique_ptr<iMesh> getMesh() const 
    {
        auto result = std::unique_ptr<iMesh>(getMesh__Raw());
        return result;
    }
};
#endif

#ifndef PATHENGINE_iTerrainCallBack_DEFINED
#define PATHENGINE_iTerrainCallBack_DEFINED
class iTerrainCallBack
{
public:
    virtual ~iTerrainCallBack() {}
    // This method is called by PathEngine to obtain the height of the terrain layer corresponding to this callback at specified x and y coordinates. 
    virtual float getHeightAt(int32_t x, int32_t y) = 0;
    // An alternative version of the height query taking extra float parameters for precision components of position. 
    virtual float getHeightAtWithPrecision(int32_t x, int32_t y, float precisionX, float precisionY) = 0;
};
#endif

#ifndef PATHENGINE_iRender3DLinesCallBack_DEFINED
#define PATHENGINE_iRender3DLinesCallBack_DEFINED
class iRender3DLinesCallBack
{
public:
    virtual ~iRender3DLinesCallBack() {}
    // This method is called by PathEngine at the start of each set of connected line segments to be rendered. 
    virtual void startVertex(int32_t terrainLayer, float x, float y, float z) = 0;
    // This method is called by PathEngine to continue a connected line sequence. 
    virtual void vertex(float x, float y, float z) = 0;
};
#endif

#ifndef PATHENGINE_iQueryCallBack_DEFINED
#define PATHENGINE_iQueryCallBack_DEFINED
class iQueryCallBack
{
public:
    virtual ~iQueryCallBack() {}
    // This method is called by PathEngine at the start of each query to determine how often the abort() method should be polled. 
    virtual int32_t desiredCallBackFrequency() = 0;
    // PathEngine calls back to this method during a pathfinding query to determine whether the query should be aborted. 
    virtual bool abort() = 0;
};
#endif

#ifndef PATHENGINE_iContentChunk_DEFINED
#define PATHENGINE_iContentChunk_DEFINED
class iContentChunk
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iContentChunk* addExternalRef__Raw() = 0;
    virtual const iContentChunk* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Validates the content chunk in advance for placement by iContentChunk::instantiate(). 
    virtual bool preValidate(int32_t maximumScale, int32_t translationRange) = 0;
    // Saves the content chunk to persistent storage. 
    virtual void save(iOutputStream& outputStream) const = 0;
    virtual iContentChunkInstance* instantiate__Raw(int32_t rotation, int32_t scale, int32_t translationX, int32_t translationY, float translationZ, int32_t sectionID) = 0;
    virtual iContentChunkInstance* instantiate_FloatRotateAndScale__Raw(float rotation, float scale, int32_t translationX, int32_t translationY, float translationZ, int32_t sectionID) = 0;
    // Used to query whether or not a content chunk has a ground component. 
    virtual bool hasGroundComponent() const = 0;
    // Used to query whether or not a content chunk has an anchors and shapes component. 
    virtual bool hasAnchorsAndShapesComponent() const = 0;
    // Used to query the number of connecting edges marked up in a chunk's ground geometry. 
    virtual int32_t numberOfConnectingEdges() const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iContentChunk> addExternalRef()
    {
        auto result = std::unique_ptr<iContentChunk>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iContentChunk> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iContentChunk>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iContentChunk* afterCast = static_cast<iContentChunk*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Used to place instances of content chunk geometry into the world. The instance objects generated can then be used to drive PathEngine's content processing functionality, and to place anchors and shapes against the resulting ground mesh. 
    std::unique_ptr<iContentChunkInstance> instantiate(int32_t rotation, int32_t scale, int32_t translationX, int32_t translationY, float translationZ, int32_t sectionID)
    {
        auto result = std::unique_ptr<iContentChunkInstance>(instantiate__Raw(rotation, scale, translationX, translationY, translationZ, sectionID));
        return result;
    }
    // Used to place instances of content chunk geometry into the world.
    // The instance objects generated can then be used to drive PathEngine's content processing functionality, and to place anchors and shapes against the resulting ground mesh. 
    std::unique_ptr<iContentChunkInstance> instantiate_FloatRotateAndScale(float rotation, float scale, int32_t translationX, int32_t translationY, float translationZ, int32_t sectionID)
    {
        auto result = std::unique_ptr<iContentChunkInstance>(instantiate_FloatRotateAndScale__Raw(rotation, scale, translationX, translationY, translationZ, sectionID));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iContentChunkInstance_DEFINED
#define PATHENGINE_iContentChunkInstance_DEFINED
class iContentChunkInstance
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iContentChunkInstance* addExternalRef__Raw() = 0;
    virtual const iContentChunkInstance* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    virtual iContentChunk* getContentChunk__Raw() const = 0;
    // A helper function to obtain the content chunk for which this is an instance. 
    virtual iContentChunk& refContentChunk() const = 0;
    // Used to build a transformed iFaceVertexMesh corresponding to the ground part (if any) of this content chunk instance. 
    virtual void buildGround(iFaceVertexBuilder& builder) const = 0;
    // Adds the set of anchors and shapes associated with this content chunk instance to the specified mesh. 
    virtual void addAnchorsAndShapes(iMesh& mesh, const char* idPrefix, int32_t numberOfTerrainLayers) const = 0;
    // Queries the horizontal range covered by the instanced geometry. 
    virtual void getHorizontalRange(cHorizontalRange& result) const = 0;
    // Used to query the number of connecting edges marked up in the instanced chunk's ground geometry. 
    virtual int32_t numberOfConnectingEdges() const = 0;
    // Used to query the number of connecting edges marked up in the instanced chunk's ground geometry. 
    virtual void getConnectingEdgeGeometry(int32_t connectingEdgeIndex, int32_t& startX, int32_t& startY, float& startZ, int32_t& endX, int32_t& endY, float& endZ) const = 0;
    // Used to check whether edge pieces can actually be connected, before actually calling iContentChunkInstance::makeEdgeConnection() for the connection. 
    virtual bool edgesCanConnect(int32_t connectingEdgeOnThis, iContentChunkInstance& targetInstance, int32_t connectingEdgeOnTarget) const = 0;
    // Used to explicitly setup bridging geometry between edge to edge connection markers.
    // This can be useful where it is desirable for a custom editor to explicitly manage these edge to edge connections, and is required when calling iMeshFederation::buildTileMeshFromContent(). 
    virtual void makeEdgeConnection(int32_t connectingEdgeOnThis, iContentChunkInstance& targetInstance, int32_t connectingEdgeOnTarget) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iContentChunkInstance> addExternalRef()
    {
        auto result = std::unique_ptr<iContentChunkInstance>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iContentChunkInstance> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iContentChunkInstance>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iContentChunkInstance* afterCast = static_cast<iContentChunkInstance*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // A helper function to obtain the content chunk for which this is an instance. 
    std::unique_ptr<iContentChunk> getContentChunk() const 
    {
        auto result = std::unique_ptr<iContentChunk>(getContentChunk__Raw());
        return result;
    }
};
#endif

#ifndef PATHENGINE_iMeshFederation_DEFINED
#define PATHENGINE_iMeshFederation_DEFINED
class iMeshFederation
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iMeshFederation* addExternalRef__Raw() = 0;
    virtual const iMeshFederation* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Saves the mesh federation object to persistent storage. 
    virtual void save(const char* format, iOutputStream& outputStream) const = 0;
    // Returns the number of meshes in this federation. 
    virtual int32_t size() const = 0;
    // Checks whether a given point is inside the range of this federation. 
    virtual bool coordinatesAreInsideRange(int32_t x, int32_t y) const = 0;
    // Looks up the tile in this federation to use for a given query, based on specified horizontal query range. 
    virtual int32_t tileForQueryRange(const cHorizontalRange& queryRange) const = 0;
    // Looks up the tile in this federation to use for a given query, based on horizontal coordinates for the centre of the query range. 
    virtual int32_t tileForQueryCentre(int32_t queryCentreX, int32_t queryCentreY) const = 0;
    // Queries the tile centre, or origin, for a specified tile in this federation. 
    virtual void getTileCentre(int32_t tileIndex, int32_t& centreX, int32_t& centreY) const = 0;
    // Queries the 'represented region' for a specified tile in this federation. 
    virtual void getRepresentedRegion_World(int32_t tileIndex, cHorizontalRange& result) const = 0;
    // Queries the 'represented region' for a specified tile in this federation. 
    virtual void getRepresentedRegion_Local(int32_t tileIndex, cHorizontalRange& result) const = 0;
    // Translates positions between the overlapping sections of adjacent meshes in this federation. 
    virtual cPosition translatePosition(int32_t fromTileIndex, const iMesh& fromMesh, int32_t toTileIndex, const iMesh& toMesh, const cPosition& fromPosition) const = 0;
    // Can be used to obtain the size requirement for the output data of iMeshFederation::getRepresentedRegionsOverlapped(). 
    virtual int32_t getNumberOfRepresentedRegionsOverlapped(const cHorizontalRange& queryRegion) const = 0;
    // Queries the set of tile represented regions overlapped by the specific horizontal region. 
    virtual void getRepresentedRegionsOverlapped(const cHorizontalRange& queryRegion, iArrayReceiver<int32_t>& results) const = 0;
    virtual iMesh* buildTileMeshFromContent__Raw(int32_t tileIndex, const iContentChunkInstance* const * chunks_Buffer, uint32_t chunks_BufferEntries, const char *const* options, iOutputStream& os) const = 0;
    // Returns the number of ground tiles overlapped by the represented region for a given federation tile. 
    virtual int32_t getNumberOfGroundTilesOverlapped(int32_t tileIndex, const iSourceTiling& tiling) const = 0;
    // Returns the set of ground tiles overlapped by the represented region for a given federation tile. 
    virtual void getGroundTilesOverlapped(int32_t tileIndex, const iSourceTiling& tiling, iArrayReceiver<int32_t>& results) const = 0;
    virtual iMesh* buildTileMeshFromLocalGroundTiles__Raw(int32_t tileIndex, const iSourceTiling& tiling, const iMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options) const = 0;
    // Translates a position from the source tiling used to construct the federation to a federation tile mesh. 
    virtual cPosition translatePositionFromSourceTile(const cPosition& fromPosition, const iSourceTiling& sourceTiling, const iMesh& sourceTile, const iMesh& federationTile) const = 0;
    // Looks up the source tile index for a position in a mesh federation. 
    virtual int32_t getSourceTileForPosition(int32_t federationTileIndex, const cPosition& position, const iSourceTiling& sourceTiling) const = 0;
    // Translates a position in a federation tile mesh back to the source tiling used to construct the federation. 
    virtual cPosition translatePositionToSourceTile(const cPosition& fromPosition, const iMesh& federationTile, const iMesh& sourceTile) const = 0;
    virtual iMesh* build2DTileMesh__Raw(int32_t tileIndex, int32_t partitioningTileSize, const char *const* options) const = 0;
    virtual const iGround* build2DTiledGround__Raw(int32_t partitioningTileSize, const char *const* options) const = 0;
    virtual iMesh* build2DTileMeshWithGround__Raw(const iGround& ground, int32_t tileIndex, const char *const* options) const = 0;
    // Queries the set of tile represented regions overlapped by the specific horizontal region. 
    virtual void getRepresentedRegionOverlappedRange(const cHorizontalRange& queryRegion, int32_t& minTileX, int32_t& minTileY, int32_t& maxTileX, int32_t& maxTileY) const = 0;
    // Returns the number of tiles across the federation, in the x direction. 
    virtual int32_t tilesInX() const = 0;
    // Returns the number of tiles in the federation, in the y direction. 
    virtual int32_t tilesInY() const = 0;
    virtual iMesh* buildTileMeshFromLocalGroundTiles_IGround__Raw(int32_t tileIndex, const iSourceTiling& tiling, const iGround* const * groundTiles_Buffer, uint32_t groundTiles_BufferEntries, const char *const* options) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iMeshFederation> addExternalRef()
    {
        auto result = std::unique_ptr<iMeshFederation>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iMeshFederation> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iMeshFederation>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iMeshFederation* afterCast = static_cast<iMeshFederation*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Generates an iMesh object for the specified tile in this federation, from local 2D content processing source data. 
    std::unique_ptr<iMesh> buildTileMeshFromContent(int32_t tileIndex, const iContentChunkInstance* const * chunks_Buffer, uint32_t chunks_BufferEntries, const char *const* options, iOutputStream& os) const 
    {
        auto result = std::unique_ptr<iMesh>(buildTileMeshFromContent__Raw(tileIndex, chunks_Buffer, chunks_BufferEntries, options, os));
        return result;
    }
    // Generates an iMesh object for the specified tile in this federation, from local ground mesh tiles. 
    std::unique_ptr<iMesh> buildTileMeshFromLocalGroundTiles(int32_t tileIndex, const iSourceTiling& tiling, const iMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const char *const* options) const 
    {
        auto result = std::unique_ptr<iMesh>(buildTileMeshFromLocalGroundTiles__Raw(tileIndex, tiling, meshes_Buffer, meshes_BufferEntries, options));
        return result;
    }
    // Generates an iMesh object with simple 2D square bounds for the specified tile in this federation. 
    std::unique_ptr<iMesh> build2DTileMesh(int32_t tileIndex, int32_t partitioningTileSize, const char *const* options) const 
    {
        auto result = std::unique_ptr<iMesh>(build2DTileMesh__Raw(tileIndex, partitioningTileSize, options));
        return result;
    }
    // Generates an iMesh object with simple 2D square bounds for the specified tile in this federation. 
    std::unique_ptr<const iGround> build2DTiledGround(int32_t partitioningTileSize, const char *const* options) const 
    {
        auto result = std::unique_ptr<const iGround>(build2DTiledGround__Raw(partitioningTileSize, options));
        return result;
    }
    // Generates an iMesh object with simple 2D square bounds for the specified tile in this federation (instanced ground version). 
    std::unique_ptr<iMesh> build2DTileMeshWithGround(const iGround& ground, int32_t tileIndex, const char *const* options) const 
    {
        auto result = std::unique_ptr<iMesh>(build2DTileMeshWithGround__Raw(ground, tileIndex, options));
        return result;
    }
    // Generates an iMesh object for the specified tile in this federation, from local ground mesh tiles. 
    std::unique_ptr<iMesh> buildTileMeshFromLocalGroundTiles_IGround(int32_t tileIndex, const iSourceTiling& tiling, const iGround* const * groundTiles_Buffer, uint32_t groundTiles_BufferEntries, const char *const* options) const 
    {
        auto result = std::unique_ptr<iMesh>(buildTileMeshFromLocalGroundTiles_IGround__Raw(tileIndex, tiling, groundTiles_Buffer, groundTiles_BufferEntries, options));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iSourceTiling_DEFINED
#define PATHENGINE_iSourceTiling_DEFINED
class iSourceTiling
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iSourceTiling* addExternalRef__Raw() = 0;
    virtual const iSourceTiling* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Returns the number of tiles in this tiling. 
    virtual int32_t size() const = 0;
    // Queries the horizontal range of an individual tile in this tiling. 
    virtual void getTileRange(int32_t tileIndex, cHorizontalRange& result) const = 0;
    // Queries the horizontal range of an individual tile in this tiling, as a string. 
    virtual const char* getTileRangeAsString(int32_t tileIndex) const = 0;
    // Queries the horizontal range to use when filtering source elements for tile by tile 3D content processing. 
    virtual void getTileFilterRange(int32_t tileIndex, cHorizontalRange& result) const = 0;
    // Saves this object to persistent storage. 
    virtual void save(const char* format, iOutputStream& outputStream) const = 0;
    // Queries the horizontal range covered by this tiling. 
    virtual void getWorldRange(cHorizontalRange& result) const = 0;
    // Queries the horizontal range to use when filtering source elements for tile by tile 3D content processing (when the 'voxels' method is being used). 
    virtual void getTileFilterRange_Voxels(int32_t tileIndex, int32_t voxelSize, cHorizontalRange& result) const = 0;
    // Queries the region covered by a specified tile in this tiling. 
    virtual void getTileRegion(int32_t tileIndex, cHorizontalRange& result) const = 0;
    // Looks up the source tile index for the specified horizontal coordinates. 
    virtual int32_t getTileForXY(int32_t x, int32_t y) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iSourceTiling> addExternalRef()
    {
        auto result = std::unique_ptr<iSourceTiling>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iSourceTiling> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iSourceTiling>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iSourceTiling* afterCast = static_cast<iSourceTiling*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

#ifndef PATHENGINE_iAnchorsAndPinnedShapes_DEFINED
#define PATHENGINE_iAnchorsAndPinnedShapes_DEFINED
class iAnchorsAndPinnedShapes
{
public:
    virtual ~iAnchorsAndPinnedShapes() {}
    // Returns the number of anchors provided by this interface. 
    virtual int32_t numberOfAnchors() const = 0;
    // Returns the name of the specified anchor, as a 'C' string. 
    virtual const char* anchor_Name(int32_t anchorIndex) const = 0;
    // Returns the horizontal coordinates of the specified anchor. 
    virtual void anchor_HorizontalPosition(int32_t anchorIndex, int32_t& x, int32_t& y) const = 0;
    // Returns the vertical range of the specified anchor. 
    virtual void anchor_VerticalRange(int32_t anchorIndex, float& z_Start, float& z_End) const = 0;
    // Returns the facing direction for the specified anchor. 
    virtual int32_t anchor_Orientation(int32_t anchorIndex) const = 0;
    // Used to query attributes of the specified anchor in an extensible manner. 
    virtual int32_t anchor_Attribute(int32_t anchorIndex, int32_t attributeIndex) const = 0;
    // Returns the number of pinned shapes provided by this interface. 
    virtual int32_t numberOfPinnedShapes() const = 0;
    // Returns the name of the specified shape, as a 'C' string. 
    virtual const char* pinnedShape_Name(int32_t shapeIndex) const = 0;
    // Returns an index for the anchor that pins the specified shape to the ground geometry. 
    virtual int32_t pinnedShape_AnchorIndex(int32_t shapeIndex) const = 0;
    // Returns the number of vertices on the specified shape. 
    virtual int32_t pinnedShape_NumberOfVertices(int32_t shapeIndex) const = 0;
    // Returns coordinates of the specified shape vertex. 
    virtual void pinnedShape_Vertex(int32_t shapeIndex, int32_t vertexIndex, int32_t& x, int32_t& y) const = 0;
    // Used to query attributes of the specified shape in an extensible manner. 
    virtual int32_t pinnedShape_Attribute(int32_t shapeIndex, int32_t attributeIndex) const = 0;
};
#endif

#ifndef PATHENGINE_iSolidObjects_DEFINED
#define PATHENGINE_iSolidObjects_DEFINED
class iSolidObjects
{
public:
    virtual ~iSolidObjects() {}
    // Returns the number of convex solids. 
    virtual int32_t numberOfConvexSolids() const = 0;
    // Returns the number of points in a convex solid object. 
    virtual int32_t convexSolid_Points(int32_t convexSolidIndex) const = 0;
    // Returns the coordinates of a point in a convex solid object. 
    virtual void convexSolid_Point(int32_t convexSolidIndex, int32_t pointIndex, int32_t& x, int32_t& y, int32_t& z) const = 0;
    // Returns the number of points in a convex solid object. 
    virtual int32_t convexSolid_Attribute(int32_t convexSolidIndex, int32_t attributeIndex) const = 0;
};
#endif

#ifndef PATHENGINE_iMeshFederationTileGeneratedCallBack_DEFINED
#define PATHENGINE_iMeshFederationTileGeneratedCallBack_DEFINED
class iMeshFederationTileGeneratedCallBack
{
public:
    virtual ~iMeshFederationTileGeneratedCallBack() {}
    // This method is called by PathEngine to pass meshes generated during the process of building a mesh federation back to the calling application. 
    virtual void tileGenerated(int32_t tileIndex, iMesh* tileMesh) = 0;
};
#endif

#ifndef PATHENGINE_iConnectionTestCallBack_DEFINED
#define PATHENGINE_iConnectionTestCallBack_DEFINED
class iConnectionTestCallBack
{
public:
    virtual ~iConnectionTestCallBack() {}
    // PathEngine calls back to this method for application-side validation of jump type connection candidates. 
    virtual void testJumpCandidatePair(int32_t fromX, int32_t fromY, int32_t fromZ, int32_t toX, int32_t toY, int32_t toZ, int32_t& penaltyOut, int32_t& penaltyBack) = 0;
    // PathEngine calls back to this method for application-side validation of drop type connection candidates. 
    virtual void testDropCandidatePair(int32_t fromX, int32_t fromY, int32_t fromZ, int32_t toX, int32_t toY, int32_t toZ, int32_t& penaltyOut, int32_t& penaltyBack) = 0;
};
#endif

#ifndef PATHENGINE_iProgressCallBack_DEFINED
#define PATHENGINE_iProgressCallBack_DEFINED
class iProgressCallBack
{
public:
    virtual ~iProgressCallBack() {}
    // This method is called by PathEngine to report progress. 
    virtual void updateProgress(const char* operationDescription, float percent) = 0;
};
#endif

#ifndef PATHENGINE_iErrorHandler_DEFINED
#define PATHENGINE_iErrorHandler_DEFINED
class iErrorHandler
{
public:
    virtual ~iErrorHandler() {}
    // This method is called by PathEngine in the case of an error to perform application side error handling, normally by displaying or logging the error. 
    virtual int32_t handle(const char* type, const char* description, const char *const* attributes) = 0;
};
#endif

#ifndef PATHENGINE_iContentProcessing3D_DEFINED
#define PATHENGINE_iContentProcessing3D_DEFINED
class iContentProcessing3D
{
public:
    // Obtains a set of attributes for the object that provides this interface.
    virtual const char *const* getVersionAttributes() const = 0;
    // Returns the major version number for the code supporting this interface. 
    virtual int32_t getInterfaceMajorVersion() const = 0;
    // Returns the minor version number for the code supporting this interface. 
    virtual int32_t getInterfaceMinorVersion() const = 0;
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iContentProcessing3D* addExternalRef__Raw() = 0;
    virtual const iContentProcessing3D* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    virtual iMesh* buildGround__Raw(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, iProgressCallBack* progressCallBack) const = 0;
    // Runs the 3D content processing functionality on a specified set of 3D source data and saves out the resulting ground mesh. 
    virtual void buildAndSaveGround(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, iProgressCallBack* progressCallBack, const char* format, bool includeMapping, iOutputStream& outputStream) const = 0;
    // Provided for generating snapshots of the 3D source data when reporting bugs in the 3D content processing functionality. 
    virtual void saveSnapshot(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, const char* format, iOutputStream& outputStream) const = 0;
    // Generates convex hulls for a set of solid objects defined by point sets (for visualisation purposes). 
    virtual void buildHullMeshesForVisualisation(const iSolidObjects& solidObjects, iFaceVertexBuilder& builder) const = 0;
    // Instantiate a content snapshot that was previously saved out to persistant storage. 
    virtual void loadSnapshot(const char* format, char const * data_Buffer, uint32_t data_BufferEntries, iFaceVertexBuilder& faceVertexBuilder, iSolidObjectsBuilder& solidObjectsBuilder, int32_t& extrudeHeight, int32_t& maxStepHeight, int32_t& maxSlope, iArrayReceiver<const char*>& options) const = 0;
    // A helper function to access the non 3D content processing parts of the PathEngine runtime. 
    virtual iPathEngine& refPathEngine() const = 0;
    virtual iPathEngine* getPathEngine__Raw() const = 0;
    virtual const iGround* buildGround_IGround__Raw(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, iProgressCallBack* progressCallBack) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iContentProcessing3D> addExternalRef()
    {
        auto result = std::unique_ptr<iContentProcessing3D>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iContentProcessing3D> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iContentProcessing3D>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iContentProcessing3D* afterCast = static_cast<iContentProcessing3D*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Runs the 3D content processing functionality on a specified set of 3D source data and builds a ground mesh directly, (with no save and load from ground mesh persistence). 
    std::unique_ptr<iMesh> buildGround(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, iProgressCallBack* progressCallBack) const 
    {
        auto result = std::unique_ptr<iMesh>(buildGround__Raw(meshes_Buffer, meshes_BufferEntries, solidObjects, extrudeHeight, maxStepHeight, maxSlope, options, progressCallBack));
        return result;
    }
    // A helper function to access the non 3D content processing parts of the PathEngine runtime. 
    std::unique_ptr<iPathEngine> getPathEngine() const 
    {
        auto result = std::unique_ptr<iPathEngine>(getPathEngine__Raw());
        return result;
    }
    // Runs the 3D content processing functionality on a specified set of 3D source data and builds a ground mesh directly, (with no save and load from ground mesh persistence). 
    std::unique_ptr<const iGround> buildGround_IGround(const iFaceVertexMesh* const * meshes_Buffer, uint32_t meshes_BufferEntries, const iSolidObjects* solidObjects, int32_t extrudeHeight, int32_t maxStepHeight, int32_t maxSlope, const char *const* options, iProgressCallBack* progressCallBack) const 
    {
        auto result = std::unique_ptr<const iGround>(buildGround_IGround__Raw(meshes_Buffer, meshes_BufferEntries, solidObjects, extrudeHeight, maxStepHeight, maxSlope, options, progressCallBack));
        return result;
    }
};
#endif

#ifndef PATHENGINE_iTestBed_DEFINED
#define PATHENGINE_iTestBed_DEFINED
class iTestBed
{
public:
    // Obtains a set of attributes for the object that provides this interface.
    virtual const char *const* getVersionAttributes() const = 0;
    // Returns the major version number for the code supporting this interface. 
    virtual int32_t getInterfaceMajorVersion() const = 0;
    // Returns the minor version number for the code supporting this interface. 
    virtual int32_t getInterfaceMinorVersion() const = 0;
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iTestBed* addExternalRef__Raw() = 0;
    virtual const iTestBed* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Report an error. 
    virtual int32_t handleError(const char* type, const char* description, const char *const* attributes) = 0;
    // Report an error. 
    virtual void reportError(const char* type, const char* errorString) = 0;
    // Used to query testbed command line arguments. 
    virtual int32_t getNumberOfCommandLineArguments() const = 0;
    // Used to query testbed command line arguments. 
    virtual const char* getCommandLineArgument(int32_t index) const = 0;
    // Used to obtain messages about any keys that went up or down since the last frame. 
    virtual const char* receiveKeyMessage() = 0;
    // Tells you whether or not a specified key is down. 
    virtual bool getKeyState(const char* keyString) = 0;
    // Tells you whether or not the left mouse button is down. 
    virtual bool getLeftMouseState() = 0;
    // Tells you whether or not the right mouse button is down. 
    virtual bool getRightMouseState() = 0;
    // Retrieves the current position of the mouse within the testbed window. 
    virtual void getMouseScreenPosition(int32_t& screenX, int32_t& screenY) = 0;
    // Retrieves a  movement delta for the mouse in screen coordinates. 
    virtual void getMouseScreenDelta(int32_t& screenDX, int32_t& screenDY) = 0;
    virtual iRenderGeometry* newRenderGeometry__Raw() const = 0;
    // Processes image file contents, to prepare for rendering. 
    virtual int32_t createImage(char const * imageData_Buffer, uint32_t imageData_BufferEntries) = 0;
    // Displays a 2D image in front of the scene, and centred in the viewport. 
    virtual void displaySplashImage_Centred(int32_t image) const = 0;
    // Displays a 2D image in front of the scene, touching the bottom left corner of the viewport. 
    virtual void displaySplashImage_BottomLeft(int32_t image) const = 0;
    // Draws a line in screen space. 
    virtual void drawLine2D(int32_t startX, int32_t startY, int32_t endX, int32_t endY) const = 0;
    // Sets colour for rendered elements that do not have their own colour information, such as render geometry and text, by colour name. 
    virtual void setColour(const char* colour) = 0;
    // Sets colour for rendered elements that do not have their own colour information, such as render geometry and text. 
    virtual void setColourRGB(float r, float g, float b) = 0;
    // Sets the background colour, by colour name. 
    virtual void setBackgroundColour(const char* colour) = 0;
    // Sets the background colour. 
    virtual void setBackgroundColourRGB(float r, float g, float b) = 0;
    // Prints a horizontal line of text at a y position that is incremented with each call in a given frame. 
    virtual void printTextLine(int32_t x, const char* text) = 0;
    // Render geometry such as lines and polygons. 
    virtual void render(const iRenderGeometry& geometry) = 0;
    // Render geometry such as lines and polygons, with a world offset. 
    virtual void render_Offset(const iRenderGeometry& geometry, float x, float y, float z) = 0;
    // Process application events and advance to the next rendering frame. 
    virtual void update(int32_t minimumFrameTime, bool& windowClosed) = 0;
    // Switches to additive blending rendering mode. 
    virtual void enterAdditiveBlendingPhase() = 0;
    // Sets the multiplier value to be applied in additive blending mode. 
    virtual void setAdditiveBlendingAlpha(float value) = 0;
    // Loads the contents of a wav file into memory to be later played by iTestBed::playSound(). 
    virtual int32_t createSound(char const * soundData_Buffer, uint32_t soundData_BufferEntries, int32_t numberOfBuffers) = 0;
    // Plays a sound. 
    virtual void playSound(int32_t soundIndex) = 0;
    // Set control state for built in testbed controls, based on mouse movement. 
    virtual void setCameraControlState(int32_t state) = 0;
    // Supply a target position that can be used to adjust camera target distance during camera control interactions, for more intuitive behaviour. 
    virtual void setCameraControlTarget(float x, float y, float z) = 0;
    // Points the camera at a target. Can be used for a chase camera. 
    virtual void lookAt(float targetX, float targetY, float targetZ, float angle, float elevation, float distance) = 0;
    // Queries the current camera position. 
    virtual void getCameraPosition(float& x, float& y, float& z) const = 0;
    // Saves current camera position and orientation. 
    virtual void saveCameraPosition(iOutputStream& outputStream) const = 0;
    // Loads camera position and orientation back from persistence. 
    virtual void loadCameraPosition(char const * data_Buffer, uint32_t data_BufferEntries) = 0;
    // Project a position in world coordinates into screen space, based on the current camera position and orientation. 
    virtual void worldToScreen(float worldX, float worldY, float worldZ, float& screenX, float& screenY) const = 0;
    // Reverse project a position to world coordinates from screen space, based on the current camera position and orientation. 
    virtual void screenToWorld(float screenX, float screenY, float& worldX, float& worldY, float& worldZ) const = 0;
    // Position the camera at a position zoomed out enough to make a given set of points visible. 
    virtual void zoomExtents(float const * worldPoints_Buffer, uint32_t worldPoints_BufferEntries) = 0;
    // Switch to 2D rendering mode. 
    virtual void enter2DMode() = 0;
    // Switch back 2D rendering mode. 
    virtual void exit2DMode() = 0;
    // Query the current rendering viewport size. 
    virtual void getWindowSize(int32_t& width, int32_t& height) const = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iTestBed> addExternalRef()
    {
        auto result = std::unique_ptr<iTestBed>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iTestBed> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iTestBed>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iTestBed* afterCast = static_cast<iTestBed*>(voidPointer);
            afterCast->release__Raw();
        }
    }
    // Creates an Interface iRenderGeometry object (which can be used for rendering lines and polygons). 
    std::unique_ptr<iRenderGeometry> newRenderGeometry() const 
    {
        auto result = std::unique_ptr<iRenderGeometry>(newRenderGeometry__Raw());
        return result;
    }
};
#endif

#ifndef PATHENGINE_iRenderGeometry_DEFINED
#define PATHENGINE_iRenderGeometry_DEFINED
class iRenderGeometry
{
public:
    // Enables you to check whether there are any outstanding references on this object, other than the interface reference through which you are calling, and therefore whether this object will actually be deleted if you call its release method. 
    virtual bool hasRefs() const = 0;
    virtual iRenderGeometry* addExternalRef__Raw() = 0;
    virtual const iRenderGeometry* addConstExternalRef__Raw() const = 0;
    virtual void release__Raw() const = 0;
    // Adds a sequence of (connected) 3D lines. 
    virtual void addLineSequence(float const * points_Buffer, uint32_t points_BufferEntries) = 0;
    // Adds a 3D polygon. 
    virtual void addPolygon(float const * points_Buffer, uint32_t points_BufferEntries) = 0;
    // Increments the internal reference count for this API object. 
    std::unique_ptr<iRenderGeometry> addExternalRef()
    {
        auto result = std::unique_ptr<iRenderGeometry>(addExternalRef__Raw());
        return result;
    }
    // Increments the internal reference count for this API object. 
    std::unique_ptr<const iRenderGeometry> addConstExternalRef() const 
    {
        auto result = std::unique_ptr<const iRenderGeometry>(addConstExternalRef__Raw());
        return result;
    }
    // Releases the API object pointer. 
    void operator delete(void* voidPointer)
    {
        if(voidPointer)
        {
            iRenderGeometry* afterCast = static_cast<iRenderGeometry*>(voidPointer);
            afterCast->release__Raw();
        }
    }
};
#endif

static const int32_t PE_FaceAttribute_SurfaceType = 0;
static const int32_t PE_FaceAttribute_SectionID = 1;
static const int32_t PE_FaceAttribute_MarksPortal = 2;
static const int32_t PE_FaceAttribute_CP3D_ExcludeFromGroundResult = 2;
static const int32_t PE_FaceAttribute_MarksExternalShape = 3;
static const int32_t PE_FaceAttribute_CP3D_ExcludeFromSteepSlopeFilter = 3;
static const int32_t PE_FaceAttribute_MarksConnectingEdge = 4;
static const int32_t PE_FaceAttribute_UserData = 5;
static const int32_t PE_FaceAttribute_MarksTerrainPortal = 6;
static const int32_t PE_FaceAttribute_Top = 6;

static const int32_t PE_AnchorAttribute_ResolvesToTerrain = 0;
static const int32_t PE_AnchorAttribute_Top = 0;
static const int32_t PE_PinnedShapeAttribute_Type = 0;
static const int32_t PE_PinnedShapeAttribute_Top = 0;
static const int32_t PE_PinnedShapeType_BurntIn = 0;
static const int32_t PE_PinnedShapeType_NamedObstacle = 1;
static const int32_t PE_PinnedShapeType_Top = 1;

static const int32_t PE_ErrorHandler_Continue = 0;
static const int32_t PE_ErrorHandler_Abort = 1;
static const int32_t PE_ErrorHandler_Break = 2;
static const int32_t PE_ErrorHandler_ContinueAndIgnoreInFuture = 3;

static const int32_t PE_CameraControl_None = 0;
static const int32_t PE_CameraControl_Zoom = 1;
static const int32_t PE_CameraControl_Pan = 2;
static const int32_t PE_CameraControl_Rotate = 3;

#ifndef PATHENGINE_I_ALLOCATOR_DEFINED
#define PATHENGINE_I_ALLOCATOR_DEFINED
class iAllocator
{
public:
    virtual ~iAllocator() {}
    virtual void* allocate(uint32_t size) = 0;
    virtual void deallocate(void* ptr) = 0;
    virtual void* expand(void* oldPtr, uint32_t oldSize, uint32_t oldSize_Used, uint32_t newSize) = 0;
// helper
    template <class T> void
    allocate_Array(uint32_t arraySize, T*& result)
    {
        result = static_cast<T*>(allocate(sizeof(T) * arraySize));
    }
    template <class T> void
    expand_Array(T*& ptr, uint32_t oldArraySize, uint32_t oldArraySize_Used, uint32_t newArraySize)
    {
        ptr = static_cast<T*>(expand(ptr, sizeof(T) * oldArraySize, sizeof(T) * oldArraySize_Used, sizeof(T) * newArraySize));
    }
};
#endif //ndef PATHENGINE_I_ALLOCATOR_DEFINED

#endif // ends file include guard
