
#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "base/types/Error.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/CameraControl.h"
#include "base/MutexLock.h"
#include "base/Container/Vector.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "platform_common/Thread.h"
#include "platform_common/Mutex.h"
#include "externalAPI/i_pathengine.h"
#include <vector>
#include <string>
#include <sstream>
#include <memory>

using PE::vector;
using std::unique_ptr;

namespace
{

class cMeshLoadingQueue
{
    int32_t _meshIterations;
    cMutex* _queueMutexes;
    unique_ptr<iMesh>* _queueMeshes;
    int32_t _nextIndex;

    cMutex _stateMutex;
    int32_t _numberLoaded;
    bool _ready;

public:

    // initialisation to be performed before foreground thread creation
    cMeshLoadingQueue(int32_t meshIterations);

    // called by background thread to load meshes and place on queue
    void loadMeshesAndPlaceOnQueue(
            iPathEngine* pathEngine,
            const char* meshName,
            const vector<const iShape*>& baseObstacleShapes,
            iShape& agentShape, const char** pathfindPreprocessAttributes
            );

    // called by foreground thread before getting meshes from queue
    // (required for background thread to lock first queue mutex)
    void waitUntilReady();

    // called by foreground thread to obtain meshes from the queue
    // returns zero when queue is empty
    // the calling code takes responsibility for deleting the mesh
    unique_ptr<iMesh> getNext();

    // called by the foreground thread just for printing in the testbed
    int32_t getNumberLoaded() const;
    int32_t getQueuePosition() const
    {
        return _nextIndex;
    }

    // destruction after foreground thread is completed
    ~cMeshLoadingQueue();
};

class cThread : public iThread
{
    iPathEngine* _pathEngine;
    cMeshLoadingQueue& _queue;
    unique_ptr<iShape> _agentShape;
    const vector<const iShape*>& _baseObstacleShapes;
    const char** _pathfindPreprocessAttributes;

public:

    cThread(
        iPathEngine* pathEngine,
        cMeshLoadingQueue& queue,
        iShape& agentShape,
        const vector<const iShape*>& baseObstacleShapes,
        const char** pathfindPreprocessAttributes
        ) :
     _pathEngine(pathEngine),
     _queue(queue),
     _agentShape(agentShape.addExternalRef()),
     _baseObstacleShapes(baseObstacleShapes),
     _pathfindPreprocessAttributes(pathfindPreprocessAttributes)
    {
    }

    void run();
};

} // end anonymous namespace

unique_ptr<iMesh>
GenerateRandomMesh(iPathEngine* pathEngine, const char* meshName, const vector<const iShape*>& baseObstacleShapes)
{
    unique_ptr<iMesh> mesh;
    {
        std::string meshPath = "../resource/meshes/";
        meshPath += meshName;
        meshPath += ".tok";
        vector<char> buffer;
        LoadBinary(meshPath.c_str(), buffer);
        assertR(!buffer.empty());
        mesh = pathEngine->loadMeshFromBuffer("tok", &buffer[0], SizeL(buffer), 0);
    }
    unique_ptr<iCollisionContext> context(mesh->newContext());
    for(int32_t i = 0; i < static_cast<int32_t>(baseObstacleShapes.size()); i++)
    {
        cPosition randomPosition;
        do
        {
            randomPosition = mesh->generateRandomPosition();
        }
        while(randomPosition.cell == -1);
        unique_ptr<iAgent> agent(mesh->placeAgent(*baseObstacleShapes[i], randomPosition));
        context->addAgent(*agent);
    }
    mesh->burnContextIntoMesh(*context);
    return mesh;
}

cMeshLoadingQueue::cMeshLoadingQueue(int32_t meshIterations)
{
    _meshIterations = meshIterations;
    _queueMutexes = new cMutex[_meshIterations];
    _queueMeshes = new unique_ptr<iMesh>[_meshIterations];
    _nextIndex = 0;
    _numberLoaded = 0;
    _ready = false;
}

void
cMeshLoadingQueue::loadMeshesAndPlaceOnQueue(
            iPathEngine* pathEngine,
            const char* meshName, const vector<const iShape*>& baseObstacleShapes,
            iShape& agentShape, const char** unobstructedSpaceAttributes
            )
{
    assertD(_meshIterations);
    _queueMutexes[0].lock();
    {
        cMutexLock lock(_stateMutex);
        _ready = true;
    }
    for(int32_t i = 0; i < _meshIterations; ++i)
    {
        _queueMeshes[i] = GenerateRandomMesh(pathEngine, meshName, baseObstacleShapes);
        _queueMeshes[i]->generateUnobstructedSpaceFor(agentShape, true, unobstructedSpaceAttributes);
        _queueMeshes[i]->generatePathfindPreprocessFor(agentShape, 0);
        if(i + 1 < _meshIterations)
        {
            _queueMutexes[i + 1].lock();
        }
        _queueMutexes[i].unlock();

        cMutexLock lock(_stateMutex);
        _numberLoaded++;
    }
}

void
cMeshLoadingQueue::waitUntilReady()
{
  // one busy wait, just to start queue synchronisation
  // (this is necessary because on some platforms mutex lock and unlock must be in same thread)
    bool ready;
    do
    {
        cMutexLock lock(_stateMutex);
        ready = _ready;
    }
    while(!ready);
}

unique_ptr<iMesh>
cMeshLoadingQueue::getNext()
{
    assertR(_ready); // waitUntilReady() must be called before starting to query queue
    if(_nextIndex == _meshIterations)
    {
        return 0;
    }
    // the critical section is locked here as a way of waiting for the next queue item to be ready
    _queueMutexes[_nextIndex].lock();
    _queueMutexes[_nextIndex].unlock();
    return std::move(_queueMeshes[_nextIndex++]);
}

int32_t
cMeshLoadingQueue::getNumberLoaded() const
{
    cMutexLock lock(_stateMutex);
    return _numberLoaded;
}

cMeshLoadingQueue::~cMeshLoadingQueue()
{
    assertD(_nextIndex == _meshIterations);
    delete [] _queueMutexes;
    delete [] _queueMeshes;
}

void
DoPathIterations(
        iTestBed* testBed,
        const iShape& agentShape,
        int32_t pathIterations,
        const cMeshLoadingQueue& queue,
        iMesh& mesh, const cMeshRenderGeometry& meshRenderGeometry, iCollisionContext* context
        )
{
    cAgentRenderGeometry agentGeometry(*testBed, agentShape, 40);
    do
    {
        meshRenderGeometry.render(*testBed);
        testBed->setColour("orange");
        for(int32_t i = 0; i < context->getNumberOfAgents(); i++)
        {
            agentGeometry.renderAt(*testBed, mesh, context->refAgent(i).getPosition());
        }

        {
            std::ostringstream os;
            os << "meshes dispatched = " << queue.getQueuePosition();
            testBed->printTextLine(10, os.str().c_str());
        }
        {
            std::ostringstream os;
            os << "meshes loaded = " << queue.getNumberLoaded();
            testBed->printTextLine(10, os.str().c_str());
        }

        for(int32_t i = 0; i < 10; i++)
        {
            if(!pathIterations)
            {
                break;
            }
            cPosition start;
            do
            {
                start = mesh.generateRandomPosition();
            }
            while(start.cell == -1 || mesh.testPointCollision(agentShape, context, start));

            cPosition goal;
            do
            {
                goal = mesh.generateRandomPosition();
            }
            while(goal.cell == -1);

            unique_ptr<iPath> path = mesh.findShortestPath(agentShape, context, start, goal);
            testBed->setColour("green");
            if(path)
                cMeshRenderGeometry::DrawPath(*testBed, *path);

            pathIterations--;
        }

        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }
    while(pathIterations);
}

unique_ptr<iCollisionContext>
InitContext(iMesh& mesh, const vector<const iShape*>& shapes)
{
    unique_ptr<iCollisionContext> result = mesh.newContext();
    cPosition randomPosition;
    do
    {
        randomPosition = mesh.generateRandomPosition();
    }
    while(randomPosition.cell == -1);
    for(uint32_t i = 0; i < shapes.size(); i++)
    {
        unique_ptr<iAgent> placed(mesh.placeAgent(*shapes[i], randomPosition));
        result->addAgent(*placed);
    }
    return result;
}

void
DoDynamicObstacleIterations(
        iTestBed* testBed,
        iShape& agentShape,
        const vector<const iShape*>& dynamicObstacleShapes,
        int32_t dynamicObstacleIterations,
        int32_t pathIterations,
        const cMeshLoadingQueue& queue,
        iMesh& mesh, const cMeshRenderGeometry& meshRenderGeometry
        )
{
    unique_ptr<iCollisionContext> context(InitContext(mesh, dynamicObstacleShapes));
    for(int32_t i = 0; i < dynamicObstacleIterations; i++)
    {
        for(int32_t j = 0; j < context->getNumberOfAgents(); j++)
        {
            cPosition randomPosition;
            do
            {
                randomPosition = mesh.generateRandomPosition();
            }
            while(randomPosition.cell == -1);
            context->refAgent(j).moveTo(randomPosition);
        }
        DoPathIterations(testBed, agentShape, pathIterations, queue, mesh, meshRenderGeometry, context.get());
    }
}

void
cThread::run()
{
    _queue.loadMeshesAndPlaceOnQueue(_pathEngine, "mesh1", _baseObstacleShapes, *_agentShape, _pathfindPreprocessAttributes);
}

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed* testBed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    unique_ptr<iShape> shape_Square20;
    {
        int32_t array[]=
        {
            -20, 20,
            20, 20,
            20, -20,
            -20, -20,
        };
        shape_Square20 = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    unique_ptr<iShape> shape_Octagon8_20;
    {
        int32_t array[]=
        {
            8,20,
            20,8,
            20,-8,
            8,-20,
            -8,-20,
            -20,-8,
            -20,8,
            -8,20,
        };
        shape_Octagon8_20 = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    unique_ptr<iShape> shape_Swarmer;
    {
        int32_t array[]=
        {
            0,16,
            14,-8,
            -14,-8,
        };
        shape_Swarmer = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    unique_ptr<iShape> shape_Square60;
    {
        int32_t array[]=
        {
            -60,60,
            60,60,
            60,-60,
            -60,-60,
        };
        shape_Square60 = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    const char* attributes[3];
    attributes[0] = "splitWithCircumferenceBelow";
    attributes[1] = "2000";
    attributes[2] = 0;

    vector<const iShape*> baseObstacles;
    baseObstacles.push_back(shape_Square60.get());
    baseObstacles.push_back(shape_Square60.get());
    baseObstacles.push_back(shape_Square60.get());
    baseObstacles.push_back(shape_Square60.get());
    baseObstacles.push_back(shape_Square60.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Swarmer.get());
    baseObstacles.push_back(shape_Square20.get());
    baseObstacles.push_back(shape_Square20.get());
    baseObstacles.push_back(shape_Octagon8_20.get());
    baseObstacles.push_back(shape_Octagon8_20.get());
    baseObstacles.push_back(shape_Octagon8_20.get());
    baseObstacles.push_back(shape_Octagon8_20.get());
    vector<const iShape*> dynamicObstacles;
    dynamicObstacles.push_back(shape_Square60.get());
    dynamicObstacles.push_back(shape_Swarmer.get());
    dynamicObstacles.push_back(shape_Octagon8_20.get());

    cMeshLoadingQueue queue(100);

    iShape& agentShape = *shape_Swarmer;

    cThread thread(
            pathEngine,
            queue,
            agentShape,
            baseObstacles, attributes
            );

    tThreadHandle handle = StartThread(thread);

    queue.waitUntilReady();

    bool isFirst = true;
    unique_ptr<iMesh> mesh;
    while(mesh = queue.getNext())
    {
        cMeshRenderGeometry meshRenderGeometry(*testBed, *mesh);
        if(isFirst)
        {
            ZoomExtents(*testBed, *mesh);
            isFirst = false;
        }
        DoDynamicObstacleIterations(
                testBed,
                agentShape,
                dynamicObstacles,
                5, // dynamic obstacle iterations
                50, // path iterations
                queue,
                *mesh, meshRenderGeometry
                );
    }

    JoinThread(handle);
}
