
#include "project/testbedApp/Benchmark/Benchmark_DynamicObstaclesPathfinding.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "base/Container/Vector.h"
#include <time.h>
#include <sstream>
#include <memory>

using PE::vector;
using std::unique_ptr;

static void
GeneratePositions(
        iMesh* mesh,
        const iShape& agentShape,
        cPosition& startPosition,
        vector<cPosition>& obstaclePositions,
        vector<cPosition>& queryPositions
        )
{
    do
    {
        startPosition = mesh->generateRandomPosition();
    }
    while(mesh->testPointCollision(agentShape, 0, startPosition));
    for(int32_t j = 0; j < static_cast<int32_t>(obstaclePositions.size()); j++)
    {
        obstaclePositions[j] = mesh->generateRandomPosition();
    }
    for(int32_t j = 0; j < static_cast<int32_t>(queryPositions.size()); j++)
    {
        queryPositions[j] = mesh->generateRandomPosition();
    }
}

static double
AddObstacles(
        iMesh* mesh,
        const iShape& obstacleShape,
        const vector<cPosition>& obstaclePositions,
        iCollisionContext* context
        )
{
    clock_t start, finish;
    int32_t j;
    start = clock();
    for(j = 0; j < static_cast<int32_t>(obstaclePositions.size()); j++)
    {
        unique_ptr<iAgent> agent(mesh->placeAgent(obstacleShape, obstaclePositions[j]));
        context->addAgent(*agent);
    }
    finish = clock();
    return static_cast<double>(finish - start) / CLOCKS_PER_SEC;
}

static double
GenerateQueryPositions(
        iMesh* mesh,
        const iShape& agentShape,
        iCollisionContext* context,
        vector<cPosition>& queryPositions
        )
{
    clock_t start, finish;
    start = clock();
    for(int32_t j = 0; j < static_cast<int32_t>(queryPositions.size()); j++)
    {
        bool blocked = mesh->testPointCollision(agentShape, context, queryPositions[j]);
        if(blocked)
        {
            queryPositions[j].cell = -1;
        }
    }
    finish = clock();
    return static_cast<double>(finish - start) / CLOCKS_PER_SEC;
}

static double
FindPaths(
        iMesh* mesh,
        const iShape& agentShape,
        iCollisionContext* context,
        const cPosition& startPosition,
        const vector<cPosition>& queryPositions,
        vector<unique_ptr<iPath>>& pathResults
        )
{
    clock_t start, finish;
    start = clock();
    for(int32_t j = 0; j < static_cast<int32_t>(queryPositions.size()); j++)
    {
        if(queryPositions[j].cell == -1)
        {
            pathResults[j] = 0;
            continue;
        }
        pathResults[j] = mesh->findShortestPath(agentShape, context, startPosition, queryPositions[j]);
    }
    finish = clock();
    return static_cast<double>(finish - start) / CLOCKS_PER_SEC;
}

void
Benchmark_DynamicObstaclesPathfinding(
        iPathEngine* pathEngine,
        iTestBed* testBed,
        iMesh* mesh,
        const iShape& agentShape,
        const iShape& obstacleShape,
        int32_t obstacles, int32_t obstacleIterations, int32_t pathIterations,
        uint32_t randomSeed,
        std::ofstream& os
        )
{
    vector<std::string> text;
    clock_t start, finish;

    double timeToPlace = 0.0;
    double timeUpdatePathfindPreprocess = 0.0;
    double timePointCollision = 0.0;
    double timeFindPath = 0.0;
    int32_t totalPathQueries = 0;
    int32_t totalPathPoints = 0;
    int32_t totalUnreachableTargets = 0;

    vector<cPosition> obstaclePositions(obstacles);
    vector<cPosition> queryPositions(pathIterations);
    vector<unique_ptr<iPath>> pathResults(pathIterations);

    pathEngine->setRandomSeed(randomSeed);
    for(int32_t i = 0; i < obstacleIterations; i++)
    {
        cPosition startPosition;
        GeneratePositions(mesh, agentShape, startPosition, obstaclePositions, queryPositions);
        unique_ptr<iCollisionContext> context(mesh->newContext());
        timeToPlace += AddObstacles(mesh, obstacleShape, obstaclePositions, context.get());

        timePointCollision += GenerateQueryPositions(mesh, agentShape, context.get(), queryPositions);

        start = clock();
        context->updatePathfindingPreprocessFor(agentShape);
        finish = clock();
        timeUpdatePathfindPreprocess += static_cast<double>(finish - start) / CLOCKS_PER_SEC;

        timeFindPath += FindPaths(mesh, agentShape, context.get(), startPosition, queryPositions, pathResults);

        for(int32_t j = 0; j < pathIterations; j++)
        {
            if(queryPositions[j].cell == -1)
            {
                continue;
            }
            ++totalPathQueries;
            if(pathResults[j])
            {
                totalPathPoints += pathResults[j]->size();
            }
            else
            {
                ++totalUnreachableTargets;
            }
        }
    }

    if(totalPathQueries == 0)
    {
        os << "!!!Error - no path queries in dynamic obstacles pathfinding benchmark.\n";
        return;
    }

    os << "DynamicObstaclesPathfinding.placement per obstacle: " << timeToPlace / obstacleIterations / obstacles << '\n';
    os << "DynamicObstaclesPathfinding.point collision per query: " << timePointCollision / obstacleIterations / pathIterations << '\n';
    os << "DynamicObstaclesPathfinding.update pathfind preprocess: " << timeUpdatePathfindPreprocess / obstacleIterations << '\n';
    os << "DynamicObstaclesPathfinding.find path: " << timeFindPath / totalPathQueries << '\n';

    {
        std::ostringstream oss;
        oss << "benchmark stats: totalPathQueries = " << totalPathQueries << ", ";
        oss << "totalUnreachableTargets = " << totalUnreachableTargets << ", ";
        oss << "totalPathPoints = " << totalPathPoints;
        text.push_back(oss.str());
    }

    ZoomExtents(*testBed, *mesh);

    int32_t obstacleIteration = 0;

    cMeshRenderGeometry meshRenderGeometry(*testBed, *mesh);

    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    while(!testBed->getKeyState("_SPACE"))
    {
        meshRenderGeometry.render(*testBed);
        testBed->printTextLine(10, "press space for next benchmark, or to finish");
        testBed->printTextLine(15, "(see file 'benchmark_results.txt' for full benchmark results)");
        {
            uint32_t i = static_cast<uint32_t>(text.size());
            while(i > 0)
            {
                i--;
                testBed->printTextLine(15, text[i].c_str());
            }
        }
        testBed->printTextLine(10, "Dynamic obstacles pathfinding benchmark");

        if(obstacleIteration == 0)
        {
            pathEngine->setRandomSeed(randomSeed);
        }
        ++obstacleIteration;
        if(obstacleIteration == obstacleIterations)
        {
            obstacleIteration = 0;
        }

        {
            cPosition startPosition;
            GeneratePositions(mesh, agentShape, startPosition, obstaclePositions, queryPositions);
            unique_ptr<iCollisionContext> context(mesh->newContext());
            AddObstacles(mesh, obstacleShape, obstaclePositions, context.get());
            GenerateQueryPositions(mesh, agentShape, context.get(), queryPositions);
            context->updatePathfindingPreprocessFor(agentShape);
            FindPaths(mesh, agentShape, context.get(), startPosition, queryPositions, pathResults);

          // draw obstacles in context
            testBed->setColour("orange");
            auto renderGeometry = testBed->newRenderGeometry();
            for(int32_t i = 0; i != obstacles; i++)
            {
                cAgentRenderGeometry::AddCylinder(context->refAgent(i), 20, *renderGeometry);
            }
            testBed->render(*renderGeometry);

          // draw paths
            testBed->setColour("green");
            for(int32_t i = 0; i != pathIterations; i++)
            {
                if(pathResults[i])
                    cMeshRenderGeometry::DrawPath(*testBed, *pathResults[i]);
            }
        }

        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }
}
