
#include "project/testbedApp/Benchmark/Benchmark_LongPaths.h"
#include "externalAPI/i_pathengine.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "base/Container/Vector.h"
#include <time.h>
#include <sstream>

static void
FindPaths(iMesh* mesh, const iShape& shape, const PE::vector<cPosition>& points)
{
    for(uint32_t i = 0; i < points.size(); i++)
    {
        for(uint32_t j = i + 1; j < points.size(); j++)
        {
            mesh->findShortestPath(shape, 0, points[i], points[j]);
        }
    }
}

void
GetPathStats(
        iMesh* mesh, const iShape& shape,
        const PE::vector<cPosition>& points,
        int32_t& totalPaths, int32_t& totalPoints,
        int32_t& numberFailed
        )
{
    totalPaths = 0;
    totalPoints = 0;
    numberFailed = 0;
    for(uint32_t i = 0; i < points.size(); i++)
    {
        for(uint32_t j = i + 1; j < points.size(); j++)
        {
            auto path = mesh->findShortestPath(shape, 0, points[i], points[j]);
            if(path)
            {
                totalPoints += path->size();
            }
            else
            {
                numberFailed++;
            }
            totalPaths++;
        }
    }
}

void
Benchmark_LongPaths(
        iTestBed* testBed,
        iMesh* mesh,
        const iShape& shape,
        std::ofstream& os
        )
{
    clock_t start, finish;
    double  duration;
    PE::vector<std::string> text;
    PE::vector<cPosition> points(60);

    {
        uint32_t i = 0;
        while(i < points.size())
        {
            cPosition randomPoint = mesh->generateRandomPosition();
            if(randomPoint.cell != -1 && !mesh->testPointCollision(shape, 0, randomPoint))
            {
                points[i++] = randomPoint;
            }
        }
    }

    start = clock();
    FindPaths(mesh, shape, points);
    finish = clock();

    duration = static_cast<double>(finish - start) / CLOCKS_PER_SEC;

    int32_t totalPaths, totalPoints, numberFailed;
    GetPathStats(mesh, shape, points, totalPaths, totalPoints, numberFailed);

    {
        std::ostringstream os;
        os << "failed paths = " << numberFailed;
        text.push_back(os.str());
    }
    os << "(failed paths = " << numberFailed << ", total points = " << totalPoints << ")\n";
    int32_t successfulPaths = totalPaths - numberFailed;
    if(successfulPaths)
    {
        std::ostringstream os;
        os << "successful paths = " << successfulPaths;
        os << ", total points = " << totalPoints;
        os << ", average points = " << static_cast<double>(totalPoints) / successfulPaths;
        text.push_back(os.str());
    }
    {
        std::ostringstream os;
        os << "total time = " << duration << ", average time = " << duration / totalPaths;
        text.push_back(os.str());
    }
    os << "average long path query time: " << duration / totalPaths << '\n';

    ZoomExtents(*testBed, *mesh);

    int32_t point1 = 0;
    int32_t point2 = 1;

    cMeshRenderGeometry meshRenderGeometry(*testBed, *mesh);

    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    while(!testBed->getKeyState("_SPACE"))
    {
        meshRenderGeometry.render(*testBed);
        testBed->printTextLine(10, "press space for next benchmark, or to finish");
        testBed->printTextLine(15, "(see file 'benchmark_results.txt' for full benchmark results)");
        {
            uint32_t i = static_cast<uint32_t>(text.size());
            while(i > 0)
            {
                i--;
                testBed->printTextLine(15, text[i].c_str());
            }
        }
        testBed->printTextLine(10, "Long paths benchmark");

        testBed->setColour("green");
        {
            int32_t i;
            for(i = 0; i < 40; i++)
            {
                auto path = mesh->findShortestPath(shape, 0, points[point1], points[point2]);
                if(path)
                    cMeshRenderGeometry::DrawPath(*testBed, *path);
                ++point2;
                if(point2 == static_cast<int32_t>(points.size()))
                {
                    ++point1;
                    point2 = point1 + 1;
                    if(point2 == static_cast<int32_t>(points.size()))
                    {
                        point1 = 0;
                        point2 = point1 + 1;
                    }
                }
            }
        }

        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }
}
