#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "project/testbedApp/CollapsibleGroup/GetGroupRootPosition.h"
#include "project/testbedApp/CollapsibleGroup/GenerateColumnTargets.h"
#include "project/testbedApp/CollapsibleGroup/ColumnTargetsManager.h"
#include "project/testbedApp/CollapsibleGroup/FormationMovementOrder.h"
#include "project/testbedApp/CollapsibleGroup/FormationMovementParameters.h"
#include "sampleShared/SelectAndMove.h"
#include "base/types/Error.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/CameraControl.h"
#include "common/STL_Helper.h"
#include "base/Container/Vector.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <string.h>
#include <memory>

using std::unique_ptr;
using PE::vector;

static void
RunDemoWithMesh(iPathEngine* pathEngine, iTestBed* testbed, iMesh* mesh)
{
  // configuration parameters
    const int32_t numberOfAgents = 80;
    const int32_t positionResolveDistance = 1500; // distance to use when resolving clicked positions
    const int32_t agentRadius = 10;
    const int32_t agentStartPlacementDistance = 200; // range to use when placing agents initially

  // parameters for the actual group movement code supplied through the following class
  // (default values are set by the class constructor)
    cFormationMovementParameters parameters;

  // create shapes
    unique_ptr<iShape> agentShape;
    {
        int32_t array[]=
        {
            -agentRadius,agentRadius,
            agentRadius,agentRadius,
            agentRadius,-agentRadius,
            -agentRadius,-agentRadius,
        };
        agentShape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    ZoomExtents(*testbed, *mesh);  // place camera to see all of newly loaded mesh

  // generate collision and pathfinding preprocess for the agent shape
    mesh->generateUnobstructedSpaceFor(*agentShape, true, 0);
    mesh->generatePathfindPreprocessFor(*agentShape, 0);

  // iCollisionContext objects encapsulate a state of collision on a mesh - essentially this is a collection of agents
  // this context will be used for collision and will contain the agent and all swarmers
    unique_ptr<iCollisionContext> context(mesh->newContext());

  // iAgent objects are used to instantiate shapes on a mesh - either as true agents or as obstructions
    vector<unique_ptr<iAgent>> agents;
    int32_t i;
    {
        cPosition placementCentre;
        do
        {
            placementCentre = mesh->generateRandomPosition();
        }
        while(mesh->testPointCollision(*agentShape, context.get(), placementCentre));
        for(i = 0; i != numberOfAgents; ++i)
        {
            cPosition p;
            do
            {
                p = mesh->generateRandomPositionLocally(placementCentre, agentStartPlacementDistance);
            }
            while(mesh->testPointCollision(*agentShape, context.get(), p));
            agents.emplace_back(mesh->placeAgent(*agentShape, p));
        }
    }

    vector<int32_t> selectedAgents;
    vector<bool> agentSelected(numberOfAgents, false);

    cSelectAndMove selectAndMove;

  // paths
    unique_ptr<iPath> groupPath;

    cFormationMovementOrder* movementOrder = 0;
    vector<int32_t> agentIndexAssignments;

    cPosition visualisePathStart, visualisePathEnd;
    visualisePathStart.cell = -1;
    visualisePathEnd.cell = -1;
    unique_ptr<iPath> visualisePath ;
    unique_ptr<iCollisionContext> visualiseContext(mesh->newContext());

    cMeshRenderGeometry meshRenderGeometry(*testbed, *mesh, *agentShape);
    cAgentRenderGeometry agentRenderGeometry(*testbed, *agentShape, 20);

    cMeshLOSPreprocess losPreprocess(*mesh);

    bool paused = false;
    bool doSingleStep = false;
    bool exitFlag = false;
    while(!exitFlag)
    {
    // generate visualisation path and agents if updated
        if(visualisePath == 0 && visualisePathStart.cell != -1 && visualisePathEnd.cell != -1)
        {
            visualiseContext->clear();
            assertD(!mesh->testPointCollision(*agentShape, 0, visualisePathStart));
            visualisePath = mesh->findShortestPath(*agentShape, context.get(), visualisePathStart, visualisePathEnd);
            if(visualisePath && visualisePath->size() > 1)
            {
                cColumnTargetsManager ctm(
                        mesh->addExternalRef(), agentShape->addExternalRef(), context->addExternalRef(), mesh->copyPath(visualisePath.get()),
                        parameters._agentSpacing, parameters._maximumWidth,
                        parameters._groupStepDistance
                        );
                do
                {
                    int32_t added = ctm.pushFront();
                    cRowTargets& row = ctm.refRow(added);
                    for(i = row.firstColumn(); i <= row.lastColumn(); ++i)
                    {
                        auto agent = mesh->placeAgent(*agentShape, row.position(i));
                        visualiseContext->addAgent(*agent);
                    }
                    ctm.popBack(added);
                }
                while(!ctm.atEnd());
            }
        }

    // draw mesh static geometry features
        meshRenderGeometry.render(*testbed);

    // draw agents
        for(i = 0; i != numberOfAgents; ++i)
        {
            if(agentSelected[i])
            {
                testbed->setColour("orange");
            }
            else
            {
                testbed->setColour("grey");
            }
            agentRenderGeometry.renderAt(*testbed, *mesh, agents[i]->getPosition());
        }

    // movement order debug draw
        if(movementOrder)
        {
            testbed->setColour("lightblue");
            movementOrder->debugDraw(testbed, *mesh);
        }

    // draw group path (if set)
        testbed->setColour("green");
        if(groupPath)
            cMeshRenderGeometry::DrawPath(*testbed, *groupPath);

    // draw visualisation path and agents, if any
        testbed->setColour("green");
        cMeshRenderGeometry::Draw3DCross(*testbed, *mesh, visualisePathStart, 30);
        cMeshRenderGeometry::Draw3DCross(*testbed, *mesh, visualisePathEnd, 30);
        if(visualisePath)
        {
            cMeshRenderGeometry::DrawPath(*testbed, *visualisePath);
            testbed->setColour("green");
            for(int32_t i = 0; i != visualiseContext->getNumberOfAgents(); ++i)
                agentRenderGeometry.renderAt(*testbed, *mesh, visualiseContext->refAgent(i).getPosition());
        }

    // update / draw click select
        {
            bool completed;
            selectAndMove.update(*testbed, completed);
            if(completed)
            {
                vector<float> screenPositions(numberOfAgents * 2);
                for(i = 0; i != numberOfAgents; ++i)
                {
                    cPosition p = agents[i]->getPosition();
                    int32_t worldPosition[3];
                    worldPosition[0] = p.x;
                    worldPosition[1] = p.y;
                    worldPosition[2] = mesh->heightAtPosition(p);
                    testbed->worldToScreen(
                        static_cast<float>(p.x),
                        static_cast<float>(p.y),
                        mesh->heightAtPositionF(p),
                        screenPositions[i * 2], screenPositions[i * 2 + 1]);
                }
                selectAndMove.updateSelectionSet(*testbed, screenPositions, agentSelected, selectedAgents);
            }
        }


    // update movement order
        if(movementOrder && (!paused || doSingleStep))
        {
            bool completed = movementOrder->advance();
            for(int32_t i = 0; i != numberOfAgents; ++i)
            {
                int32_t indexAssignment = agentIndexAssignments[i];
                if(indexAssignment != -1)
                {
                    agents[i]->moveTo(movementOrder->getCurrentPosition(indexAssignment));
                }
            }
            if(completed)
            {
                delete movementOrder;
                movementOrder = 0;
            }
        }
        doSingleStep = false;

        CameraControl(*testbed, losPreprocess);

    // tell the testbed to render this frame
        testbed->update(40, exitFlag);

    // receive and process messages for all keys pressed since last frame
        const char* keyPressed;
        while(keyPressed = testbed->receiveKeyMessage())
        {
            {
                bool handled, moveRequested;
                selectAndMove.handleInputMessage(*testbed, keyPressed, handled, moveRequested);
                if(handled)
                {
                    if(moveRequested)
                    {
                        delete movementOrder;
                        movementOrder = 0;

                        if(selectedAgents.empty())
                        {
                            break;
                        }

                        cPosition targetPosition = losPreprocess.positionAtMouse(*testbed);

                        if(targetPosition.cell == -1)
                        {
                            break;
                        }
                        targetPosition = mesh->findClosestUnobstructedPosition(*agentShape, context.get(), targetPosition, positionResolveDistance);
                        if(targetPosition.cell == -1)
                        {
                            break;
                        }

                        vector<cPosition> positions(selectedAgents.size());
                        for(int32_t i = 0; i != SizeL(selectedAgents); ++i)
                        {
                            int32_t j = selectedAgents[i];
                            if(agents[j]->testCollisionAt(0, agents[j]->getPosition()))
                            {
                                cPosition p = agents[j]->findClosestUnobstructedPosition(0, positionResolveDistance);
                                if(p.cell != -1)
                                {
                                    agents[j]->moveTo(p);
                                }
                            }
                            positions[i] = agents[j]->getPosition();
                        }
                        vector<int32_t> indexAssignments;
                        movementOrder = new cFormationMovementOrder(
                                mesh, *agentShape, context.get(),
                                parameters,
                                positions,
                                targetPosition,
                                indexAssignments
                                );
                        agentIndexAssignments.resize(0);
                        agentIndexAssignments.resize(agents.size(), -1);
                        for(int32_t i = 0; i != SizeL(selectedAgents); ++i)
                        {
                            agentIndexAssignments[selectedAgents[i]] = indexAssignments[i];
                        }
                    }
                    continue;
                }
            }

            if(keyPressed[0] != 'd') // is it a key down message?
                continue;

            switch(keyPressed[1])
            {
            case 'S':
                visualisePathStart = losPreprocess.positionAtMouse(*testbed);
                visualisePath = nullptr;
                break;
            case 'E':
                visualisePathEnd = losPreprocess.positionAtMouse(*testbed);
                visualisePath = nullptr;
                break;
            case 'P':
                paused = !paused;
                break;
            case 'A':
                doSingleStep = true;
                break;
            case '_':
                if(!strcmp("ESCAPE", keyPressed + 2))
                {
                    exitFlag = true;
                    break;
                }
            }
        }
    }

    delete movementOrder;
}

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed* testbed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testbed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testbed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    unique_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/thainesford_town.tok", buffer);
        mesh = pathEngine->loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
    }

    RunDemoWithMesh(pathEngine, testbed, mesh.get());

    assertD(!mesh->hasRefs());
}
