#include "base/types/Header.h"
#include "project/testbedApp/CollapsibleGroup/ClusterMovementOrder.h"
#include "project/testbedApp/CollapsibleGroup/GetGroupRootPosition.h"
#include "project/testbedApp/CollapsibleGroup/ColumnTargetsManager.h"
#include "project/testbedApp/CollapsibleGroup/FormationMovementParameters.h"
#include "project/testbedApp/CollapsibleGroup/AssignTargets.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "base/Container/Vector.h"
#include "externalAPI/i_pathengine.h"
#include <math.h>
#include <algorithm>

using std::unique_ptr;
using PE::vector;

namespace
{

class cAgentInFrontPredicate
{
    const vector<unique_ptr<cCollapsibleGroupAgent>>* _agents;
public:
    cAgentInFrontPredicate(const vector<unique_ptr<cCollapsibleGroupAgent>>* agents)
    {
        _agents = agents;
    }
    bool
    operator()(int32_t lhs, int32_t rhs) const
    {
        return (*_agents)[lhs]->isInFrontOf(*(*_agents)[rhs]);
    }
};

} // anonymous namespace

static int32_t
ColumnPriority(int32_t column)
{
    if(column >= 0)
    {
        return column * 2;
    }
    return (-1 - column) * 2 + 1;
}

static double
Distance(const cPosition& p1, const cPosition& p2)
{
    double dx = static_cast<double>(p1.x) - p2.x;
    double dy = static_cast<double>(p1.y) - p2.y;
    return sqrt(dx * dx + dy * dy);
}

static void
InterpolatePosition(
        const iMesh* mesh,
        const cPosition& p1, const cPosition& p2,
        double ratio,
        cPosition& interpolatedPosition
        )
{
    double dx = static_cast<double>(p2.x) - p1.x;
    double dy = static_cast<double>(p2.y) - p1.y;
    interpolatedPosition.x = p1.x + static_cast<int32_t>(dx * ratio);
    interpolatedPosition.y = p1.y + static_cast<int32_t>(dy * ratio);
    interpolatedPosition.cell = mesh->getCellForEndOfLine(p1, interpolatedPosition.x, interpolatedPosition.y);
    if(interpolatedPosition.cell == -1)
    {
        interpolatedPosition = p1;
    }
}

double
cCollapsibleGroupAgent::getSpacingRequired(
        cColumnTargetsManager& targetsManager,
        double groupStepDistance,
        int32_t toRow,
        int32_t toColumn
        )
{
    assertD(toRow <= targetsManager.frontRowIndex());
    if(toRow >= targetsManager.frontRowIndex())
    {
        return 0.;
    }
    cRowTargets& afterRow = targetsManager.refRow(toRow + 1);
    if(!afterRow.isValidColumn(toColumn))
    {
        return 0.;
    }
    const cCollapsibleGroupAgent* agentInFront = afterRow.assignedAgent(toColumn);
    if(!agentInFront)
    {
        return 0.;
    }
    double agentInFrontTravelled = agentInFront->_currentLinkLength - agentInFront->_distanceRemaining;
    return groupStepDistance - agentInFrontTravelled;
}

bool
cCollapsibleGroupAgent::advanceStep(
        const iMesh* mesh,
        const iCollisionContext* context,
        cColumnTargetsManager& targetsManager,
        double groupStepDistance,
        double& advanceDistance,
        const cCollapsibleGroupAgent* leader,
        PE::vector<cPosition>& positionsToUpdate
        )
{
    if(_state == AT_END)
    {
        advanceDistance = 0.f;
        return false;
    }
    if(_state == MOVING_TO_START)
    {
        assertD(_path);
        {
            float precisionX, precisionY;
            _path->advanceAlong(*_agentShape, static_cast<float>(advanceDistance), context, precisionX, precisionY);
        }
        positionsToUpdate[_agentIndex] = _path->position(0);
        if(_path->size() == 1 && !targetsManager.refRow(_toRow).isAssigned(_toColumn))
        {
            _path.reset();
            targetsManager.refRow(_toRow).assign(_toColumn, this); // we depend on the address of this object remaining constant
            _state = WAITING_FOR_SLOT;
            _fromRow = _toRow;
            _fromColumn = _toColumn;
        }
        advanceDistance = 0.f;
        return true;
    }
    assertD(_path == 0);
    if(_state == MOVING_TO_SLOT)
    {
        cPosition toPosition = targetsManager.refRow(_toRow).position(_toColumn);
        if(_distanceRemaining <= advanceDistance)
        {
            _distanceRemaining = 0.f;
            advanceDistance = advanceDistance - _distanceRemaining;
            assertD(advanceDistance >= 0.f);
        }
        else
        {
            _distanceRemaining -= advanceDistance;
            advanceDistance = 0.f;
        }
        if(leader && leader->_fromRow == _fromRow && leader->_state == MOVING_TO_SLOT && _distanceRemaining < leader->_distanceRemaining)
        {
            _distanceRemaining = leader->_distanceRemaining;
            advanceDistance = 0.f;
        }
        double spacingRequired = getSpacingRequired(targetsManager, groupStepDistance, _toRow, _toColumn);
        if(_distanceRemaining < spacingRequired)
        {
            _distanceRemaining = spacingRequired;
            advanceDistance = 0.f;
        }

        if(_distanceRemaining == 0.f)
        {
            positionsToUpdate[_agentIndex] = toPosition;
            _state = WAITING_FOR_SLOT;
            _fromRow = _toRow;
            _fromColumn = _toColumn;
        }
        else
        {
            cPosition fromPosition = targetsManager.refRow(_fromRow).position(_fromColumn);
            cPosition interpolatedPosition;
            InterpolatePosition(mesh, toPosition, fromPosition, _distanceRemaining / _currentLinkLength, interpolatedPosition);
            positionsToUpdate[_agentIndex] = interpolatedPosition;
        }
        return true;
    }
    assertD(_state == WAITING_FOR_SLOT);

    if(leader && leader->_fromRow == _fromRow && leader->_state == WAITING_FOR_SLOT)
    {
        advanceDistance = 0.f;
        return false;
    }

    _toRow = _fromRow + 1;
    if(_toRow > targetsManager.frontRowIndex())
    {
        if(targetsManager.atEnd())
        {
            _state = AT_END;
            return false;
        }
        targetsManager.pushFront();
    }

    cRowTargets& fromRow = targetsManager.refRow(_fromRow);
    cRowTargets& toRow = targetsManager.refRow(_toRow);

    int32_t preferredSlots[3];
    if(_fromColumn < 0)
    {
        preferredSlots[0] = _fromColumn + 1;
        preferredSlots[1] = _fromColumn;
        preferredSlots[2] = _fromColumn - 1;
    }
    else
    if(_fromColumn > 0)
    {
        preferredSlots[0] = _fromColumn - 1;
        preferredSlots[1] = _fromColumn;
        preferredSlots[2] = _fromColumn + 1;
    }
    else
    {
        preferredSlots[0] = _fromColumn;
        preferredSlots[1] = _fromColumn + 1;
        preferredSlots[2] = _fromColumn - 1;
    }
    preferredSlots[0] = toRow.closestValidColumn(preferredSlots[0]);

    //if(_fromColumn && fromRow.centringAndWidthAreIdentical(toRow))
    //{
    //    if(fromRow.isAssigned(preferredSlots[0]) || toRow.isAssigned(preferredSlots[0]))
    //    {
    //      // this case added to try to get agents to 'lock in' to formation when no changes in corridor width
    //        // following stops agents replacing inside agent, but allows overtake
    //        preferredSlots[0] = _fromColumn;
    //        assertD(preferredSlots[1] == _fromColumn);
    //    }
    //}

    PE::vector<int32_t> overtakingSlots;
    PE::vector<int32_t> nonOvertakingSlots;
    int32_t i;
    for(i = 0; i != 3; ++i)
    {
        int32_t targetColumn = preferredSlots[i];
        if(!toRow.isValidColumn(targetColumn))
        {
            continue;
        }
        if(toRow.isAssigned(targetColumn))
        {
            continue;
        }
        double spacingRequired = getSpacingRequired(targetsManager, groupStepDistance, _toRow, targetColumn);
        if(spacingRequired > 0.)
        {
            nonOvertakingSlots.push_back(targetColumn);
        }
        else
        {
            overtakingSlots.push_back(targetColumn);
        }
    }

    PE::vector<int32_t> orderedSlots = overtakingSlots;
    for(i = 0; i != SizeL(nonOvertakingSlots); ++i)
    {
        orderedSlots.push_back(nonOvertakingSlots[i]);
    }

    if(!orderedSlots.empty())
    {
      // use best open slot
        int32_t targetColumn = orderedSlots[0];
        toRow.assign(targetColumn, this); // we depend on the address of this object remaining constant
        _toColumn = targetColumn;
        fromRow.unassign(_fromColumn, this); // we depend on the address of this object remaining constant
        _state = MOVING_TO_SLOT;
        {
            cPosition toPosition = toRow.position(_toColumn);
            cPosition fromPosition = fromRow.position(_fromColumn);
            _currentLinkLength = Distance(fromPosition, toPosition);
        }
        _distanceRemaining = _currentLinkLength;
        return true;
    }

    // didn't find a slot yet
    advanceDistance = 0.f; // can't advance for this update
    return false;
}

cCollapsibleGroupAgent::cCollapsibleGroupAgent(
        const iMesh* mesh,
        const iCollisionContext* context,
        const iShape& agentShape,
        int32_t agentIndex,
        const cPosition& agentStartPosition,
        cColumnTargetsManager& targetsManager,
        int32_t startRow, int32_t startColumn,
        bool& failed_NoPath
        ) :
    _agentShape(agentShape.addConstExternalRef())
{
    _agentIndex = agentIndex;
    _state = MOVING_TO_START;
    _toRow = startRow;
    _toColumn = startColumn;
    cPosition startPositionInFormation = targetsManager.refRow(startRow).position(startColumn);
    if(!mesh->testPointCollision(agentShape, 0, agentStartPosition))
    {
        _path = mesh->findShortestPath(agentShape, context, agentStartPosition, startPositionInFormation);
    }
    failed_NoPath = false;
    if(!_path)
    {
        _state = FAILED;
        failed_NoPath = true;
    }

}

double
cCollapsibleGroupAgent::distanceFromFrontRow(
        cColumnTargetsManager& targetsManager,
        double groupStepDistance
        ) const
{
    int32_t row;
    double distanceToRow = 0.f;
    if(_state == MOVING_TO_START)
    {
        row = _toRow;
        distanceToRow = _path->getLength();
    }
    else
    if(_state == MOVING_TO_SLOT)
    {
        row = _toRow;
        distanceToRow = _distanceRemaining;
    }
    else
    {
        row = _fromRow;
        distanceToRow = 0.f;
    }
    assertD(targetsManager.frontRowIndex() >= row);
    return distanceToRow + (targetsManager.frontRowIndex() - row) * groupStepDistance;
}

bool
cCollapsibleGroupAgent::isInFrontOf(const cCollapsibleGroupAgent& rhsAgent) const
{
    const cCollapsibleGroupAgent& lhsAgent = *this;
    bool lhsReachedStart = (lhsAgent._state != MOVING_TO_START);
    bool rhsReachedStart = (rhsAgent._state != MOVING_TO_START);
    if(lhsReachedStart != rhsReachedStart)
    {
        return lhsReachedStart;
    }
    if(!lhsReachedStart)
    {
        return _path->getLength() < rhsAgent._path->getLength();
    }
    if(_toRow != rhsAgent._toRow)
    {
        return _toRow > rhsAgent._toRow;
    }
    //return _distanceRemaining < rhsAgent._distanceRemaining;
    return ColumnPriority(_toColumn) < ColumnPriority(rhsAgent._toColumn);
}

bool
cCollapsibleGroupAgent::advance(
        const iMesh* mesh,
        const iCollisionContext* context,
        cColumnTargetsManager& targetsManager,
        double groupStepDistance,
        double advanceDistance,
        const cCollapsibleGroupAgent* leader,
        PE::vector<cPosition>& positionsToUpdate
        )
{
    if(leader == this)
    {
        leader = 0;
    }
    bool moved = false;
    assertD(advanceDistance > 0.f);
    do
    {
        if(advanceStep(mesh, context, targetsManager, groupStepDistance, advanceDistance, leader, positionsToUpdate))
        {
            moved = true;
        }
    }
    while(advanceDistance > 0.f);
    return moved;
}

void
cCollapsibleGroupAgent::retire()
{
    _path.reset();
}

void
cCollapsibleGroupAgent::debugDraw(cColumnTargetsManager& targetsManager, iTestBed* testbed, iMesh& mesh) const
{
    if(_state == MOVING_TO_START)
    {
        if(_path)
            cMeshRenderGeometry::DrawPath(*testbed, *_path);
    }
    else
    if(_state == MOVING_TO_SLOT)
    {
        cRowTargets& fromRow = targetsManager.refRow(_fromRow);
        cRowTargets& toRow = targetsManager.refRow(_toRow);
        cPosition toPos = toRow.position(_toColumn);
        cMeshRenderGeometry::DrawLineOnGround(*testbed, mesh, fromRow.position(_fromColumn), toPos.x, toPos.y);
    }
}

cClusterMovementOrder::cClusterMovementOrder(
        const iMesh* mesh,
        const iShape& agentShape,
        const iCollisionContext* context,
        const cFormationMovementParameters& parameters,
        const PE::vector<cPosition>& agentPositions,
        const cPosition& targetPosition,
        int32_t& nextIndex,
        bool& failed_NoPath,            
        PE::vector<int32_t>& indexAssignments
        ) :
 _parameters(parameters)
{
    assertD(!agentPositions.empty());
    assertD(indexAssignments.empty());

    _targetsManager = 0;
    {
        cPosition startP;
        GetGroupRootPosition(agentPositions, startP);
        if(startP.cell == -1) // should not happen unless agents are inside obstructed space for the context
        {
            failed_NoPath = true;
            return;
        }
        assertD(!mesh->testPointCollision(agentShape, 0, startP));
        auto path = mesh->findShortestPath(agentShape, context, startP, targetPosition);
        if(!path)
        {
            failed_NoPath = true;
            return;
        }
        _targetsManager = new cColumnTargetsManager(
                mesh->addConstExternalRef(),
                agentShape.addConstExternalRef(),
                context->addConstExternalRef(),
                std::move(path),
                parameters._agentSpacing, parameters._maximumWidth,
                parameters._groupStepDistance
                );
    }

    failed_NoPath = false;

    _mesh = mesh;
    _context = context;

    PE::vector<int32_t> startRows;
    PE::vector<int32_t> startColumns;
    PE::vector<cPosition> startPositions;
    do
    {
        int32_t rowIndex = _targetsManager->pushFront();
        cRowTargets& row = _targetsManager->refRow(rowIndex);
        int32_t column = row.firstColumn();
        do
        {
            startRows.push_back(rowIndex);
            startColumns.push_back(column);
            startPositions.push_back(row.position(column));
            ++column;
        }
        while(column <= row.lastColumn() && SizeL(startPositions) < parameters._numberOfStartPositions);
    }
    while(!_targetsManager->atEnd() && SizeL(startPositions) < parameters._numberOfStartPositions);


    PE::vector<int32_t> assignments;
    AssignTargets(agentPositions, startPositions, assignments);

    indexAssignments.resize(agentPositions.size());
    int32_t i;
    for(i = 0; i != SizeL(agentPositions); ++i)
    {
        int32_t j = assignments[i];
        bool noPathForAgent;
        unique_ptr<cCollapsibleGroupAgent> cga(new cCollapsibleGroupAgent(
                mesh, context,
                agentShape, nextIndex,
                agentPositions[i],
                *_targetsManager, startRows[j], startColumns[j],
                noPathForAgent
                ));
        if(noPathForAgent)
        {
            indexAssignments[i] = -1;
        }
        else
        {
            indexAssignments[i] = nextIndex++;
            _agents.push_back(std::move(cga));
        }
    }
}
cClusterMovementOrder::~cClusterMovementOrder()
{
    for(int32_t i = 0; i != SizeL(_agents); ++i)
    {
        _agents[i]->retire();
    }
    delete _targetsManager;
}

bool
cClusterMovementOrder::advance(PE::vector<cPosition>& positionsToUpdate)
{
    assertD(!_agents.empty());
    bool movedAtLeastOne = false;
    PE::vector<int32_t> agentOrder(_agents.size());
    int32_t i;
    for(i = 0; i != SizeL(_agents); ++i)
    {
        agentOrder[i] = i;
    }
    std::sort(agentOrder.begin(), agentOrder.end(), cAgentInFrontPredicate(&_agents));
    double leaderSpeed = _parameters._agentSpeed;
    double followersSpeed = leaderSpeed * 2.f;

    for(i = 0; i != SizeL(_agents); ++i)
    {
        double speed = i ? followersSpeed : leaderSpeed;
        if(speed == 0.f)
        {
            continue;
        }
        if(_agents[agentOrder[i]]->advance(_mesh, _context, *_targetsManager, _parameters._groupStepDistance, speed, &*_agents[agentOrder[0]], positionsToUpdate))
        {
            movedAtLeastOne = true;
        }
    }
    return !movedAtLeastOne;
}


void
cClusterMovementOrder::debugDraw(iTestBed* testbed, iMesh& mesh) const
{
    for(int32_t i = 0; i != SizeL(_agents); ++i)
    {
        _agents[i]->debugDraw(*_targetsManager, testbed, mesh);
    }
}
