#include "base/types/Header.h"
#include "project/testbedApp/CollapsibleGroup/ColumnTargetsManager.h"
#include "project/testbedApp/CollapsibleGroup/GenerateColumnTargets.h"

cColumnTargetsManager::cColumnTargetsManager(
        std::unique_ptr<const iMesh> mesh,
        std::unique_ptr<const iShape> shape,
        std::unique_ptr<const iCollisionContext> context,
        std::unique_ptr<iPath> path,
        int32_t agentSpacing, int32_t maximumWidth,
        double advanceStep
        )
{
    assertD(path);
    _mesh = std::move(mesh);
    _shape = std::move(shape);
    _context = std::move(context);
    _remainingPath = std::move(path);
    _agentSpacing = agentSpacing;
    _maximumWidth = maximumWidth;
    _advanceStep = advanceStep;
    _startIndex = 0;
    _currentSegmentEnd = _remainingPath->position(1);
    _currentSegmentForwardX = _currentSegmentEnd.x - _remainingPath->position(0).x;
    _currentSegmentForwardY = _currentSegmentEnd.y - _remainingPath->position(0).y;
    _atEnd = false;
    _frontRowDistance = 0.;
}

bool
cColumnTargetsManager::atEnd() const
{
    return _atEnd;
}

int32_t
cColumnTargetsManager::pushFront()
{
    assertD(!_atEnd);

    cPosition basePosition = _remainingPath->position(0);
    PE::vector<cPosition> leftTargets, rightTargets;
    GenerateColumnTargets(
            *_mesh, *_shape, _context.get(),
            basePosition,
            _currentSegmentForwardX, _currentSegmentForwardY,
            _agentSpacing, _maximumWidth,
            leftTargets, rightTargets
            );

    int32_t result = _startIndex + SizeL(_targetsWindow);
    _targetsWindow.push_back(cRowTargets(leftTargets, basePosition, rightTargets));

    _frontRowDistance += _advanceStep;
        
    if(_remainingPath->size() == 1)
    {
        _atEnd = true;
    }
    else
    {
        float precisionX, precisionY; // ignored here
        _remainingPath->advanceAlong(*_shape, static_cast<float>(_advanceStep), _context.get(), precisionX, precisionY);
        if(_remainingPath->size() > 1 && _remainingPath->position(1) != _currentSegmentEnd)
        {
          // update segment direction vector only on change of path segment
            _currentSegmentEnd = _remainingPath->position(1);
            _currentSegmentForwardX = _currentSegmentEnd.x - _remainingPath->position(0).x;
            _currentSegmentForwardY = _currentSegmentEnd.y - _remainingPath->position(0).y;
        }
    }

    // following added to remove messy looking shorter step distance position set at end of path
    if(_remainingPath->getLength() < _advanceStep)
    {
        _atEnd = true;
    }

    return result;
}
void
cColumnTargetsManager::popBack(int32_t rowIndex)
{
    _targetsWindow.pop_front();
    ++_startIndex;
}

cRowTargets&
cColumnTargetsManager::refRow(int32_t rowIndex)
{
    assertD(rowIndex >= _startIndex);
    rowIndex -= _startIndex;
    assertD(rowIndex < SizeL(_targetsWindow));
    return _targetsWindow[rowIndex];
}

int32_t
cColumnTargetsManager::frontRowIndex() const
{
    assertD(!_targetsWindow.empty());
    return _startIndex + SizeL(_targetsWindow) - 1;
}
double
cColumnTargetsManager::frontRowDistance() const
{
    return _frontRowDistance;
}
