#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "base/types/Error.h"
#include "sampleShared/LoadWhiteSpaceDelimited.h"
#include "sampleShared/LoadContentChunkPlacement.h"
#include "sampleShared/SimpleDOM.h"
#include "sampleShared/PartitionedTerrain.h"
#include "sampleShared/FaceVertexBuilder.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/CameraControl.h"
#include "common/FileOutputStream.h"
#include "base/Container/Vector.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <string>
#include <sstream>
#include <fstream>
#include <memory>

using PE::vector;
using std::unique_ptr;

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed* testBed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    cSimpleDOM placementScript;
    {
        std::ifstream is("../resource/contentChunkPlacement/contentchunks_example.txt");
        if(!is.good())
        {
            Error("Fatal", "Could not open content chunk placement script('../resource/contentChunkPlacement/contentchunks_example.txt').");
            return;
        }
        LoadWhiteSpaceDelimited(is, placementScript);
    }

    vector<unique_ptr<iContentChunkInstance>> placedInstances;
    LoadContentChunkPlacement(pathEngine, testBed, placementScript, placedInstances, false);

  // generate terrain stand-in geometry for the range -100000,-100000 -> 100000,100000
  // world coordinates are in millimetres, so this corresponds to a 200 metres by 200 metres region centred on the origin
    cPartitionedTerrain terrain(-100000, -100000, 20000, 10);

    vector<const iFaceVertexMesh*> groundParts;

    groundParts.push_back(&terrain);

    cFaceVertexBuilder faceVertexBuilder;
    for(uint32_t i = 0; i < placedInstances.size(); ++i)
    {
        placedInstances[i]->buildGround(faceVertexBuilder);
    }
    // ** faceVertexBuilder needs to stay in scope for these interface pointers to remain valid
    faceVertexBuilder.addInterfacePointers(groundParts);

    // the following lines can be uncommented to write a file containing a snapshot of the data being passed into content processing
    // this can be imported into 3DS Max, and is useful for troubleshooting any problems with the content processing
//    {
//        cFileOutputStream fos("contentSnapshot.tok");
//        pathEngine->saveContentData(&groundParts.front(), groundParts.size(), "tok", fos);
//    }

    unique_ptr<iMesh> mesh(pathEngine->buildMeshFromContent(&groundParts.front(), SizeL(groundParts), 0));

    if(mesh)
    {
        for(uint32_t  i = 0; i < placedInstances.size(); ++i)
        {
            std::ostringstream prefix;
            prefix << i << ':';
            placedInstances[i]->addAnchorsAndShapes(*mesh, prefix.str().c_str(), 1);
        }
    }
    else
    {
        return;
    }

    ZoomExtents(*testBed, *mesh);

    cMeshRenderGeometry meshRenderGeometry(*testBed, *mesh);

    bool windowClosed = false;
    while(!testBed->getKeyState("_SPACE") && !windowClosed)
    {
        meshRenderGeometry.render(*testBed);
        testBed->printTextLine(10, "press space to quit..");
        testBed->printTextLine(10, "result of buildMeshFromContent()");
        CameraControl(*testBed);
        testBed->update(20, windowClosed);
    }
}
