#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "base/types/Error.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include "sampleShared/MoveAgent.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/CameraControl.h"
#include "sampleShared/Error.h"
#include "common/STL_Helper.h"
#include "base/Container/Vector.h"
#include <cstring>
#include <cstdlib>
#include <cassert>
#include <time.h>
#include <sstream>
#include <memory>

using PE::vector;
using std::unique_ptr;

namespace
{

class cTimeTracker
{
    vector<double> _times;
    int32_t _bufferPosition;
    bool _bufferFull;
public:
    cTimeTracker() :
      _times(40),
      _bufferPosition(0),
      _bufferFull(false)
    {
    }
    void addTiming(double value)
    {
        _times[_bufferPosition++] = value;
        if(_bufferPosition == SizeL(_times))
        {
            _bufferPosition = 0;
            _bufferFull = true;
        }
    }
    bool ready() const
    {
        return _bufferFull;
    }
    double get()
    {
        assertD(ready());
        double result = _times[0];
        int32_t i;
        for(i = 1; i != SizeL(_times); ++i)
        {
            result += _times[i];
        }
        return result /= static_cast<double>(SizeL(_times));
    }
};

class cQueryCallBack : public iQueryCallBack
{
    int32_t _frequency;
    int32_t _count;

public:

    cQueryCallBack(int32_t frequency, int32_t count) :
      _frequency(frequency),
      _count(count)
    {
        assertD(_count);
    }

    int32_t desiredCallBackFrequency()
    {
        return _frequency;
    }
    bool abort()
    {
        --_count;
        return _count == 0;
    }
};

} // anonymous namespace

static cPosition
RandomUnobstructedPosition(const iShape& shape, const iMesh& mesh, const iCollisionContext* context)
{
    while(1)
    {
        cPosition candidate = mesh.generateRandomPosition();
        if(!mesh.testPointCollision(shape, context, candidate))
        {
            return candidate;
        }
    }
}

static cPosition
LocalRandomUnobstructedPosition(const iAgent& agent, const iCollisionContext* context)
{
    auto& mesh = agent.refMesh();
    for(int32_t i = 0; i != 10; ++i)
    {
        cPosition candidate = mesh.generateRandomPositionLocally(agent.getPosition(), 1000);
        if(!agent.testCollisionAt(context, candidate))
        {
            return candidate;
        }
    }
    cPosition invalidPosition;
    invalidPosition.cell = -1;
    return invalidPosition;
}

static void
FindPaths(
        const vector<unique_ptr<iAgent>>& agents, vector<unique_ptr<iPath>>& paths,
        const iCollisionContext* pathfindContext,
        int32_t pathfindingQuota)
{
    assertD(agents.size() == paths.size());
    int32_t startI = rand() % SizeL(agents);
    for(int32_t i = 0; i != SizeL(agents); ++i)
    {
        int32_t j = startI + i;
        if(j >= SizeL(agents))
        {
            j -= SizeL(agents);
        }
        auto& agent = *agents[j];
        if(paths[j])
        {
            continue;
        }
        if(pathfindingQuota-- <= 0)
        {
            return;
        }
        auto& mesh = agent.refMesh();
        unique_ptr<const iShape> shape(agent.getShape());
        for(int32_t k = 0; k != 4; ++k)
        {
            auto target = LocalRandomUnobstructedPosition(agent, pathfindContext);
            if(target.cell == -1)
            {
                continue;
            }
            cQueryCallBack callBack(5, 10);
            paths[j] = mesh.findShortestPath_WithQueryCallBack(*shape, pathfindContext, agent.getPosition(), target, &callBack);
            if(paths[j])
            {
                break;
            }
        }
    }
}

static void
AdvanceAgents(const vector<unique_ptr<iAgent>>& agents, vector<unique_ptr<iPath>>& paths, const iCollisionContext* context)
{
    assertD(agents.size() == paths.size());
    for(int32_t i = 0; i != SizeL(agents); ++i)
    {
        auto& path = paths[i];
        if(path == nullptr)
        {
            continue;
        }
        bool collided = agents[i]->advanceAlongPath(path.get(), 8.f, context);
        if(collided || path->size() == 1)
        {
            path = nullptr;
        }
    }
}

static void
CheckAndCollisionsAndRevert_Simple(
        const vector<unique_ptr<iAgent>>& agents,
        vector<unique_ptr<iPath>>& paths,
        const vector<cPosition>& startPositions,
        const iCollisionContext* checkContext
        )
{
    // simpler logic, but potentially resulting in multiple
    // collision testing passes through the full agent vector
    while(1)
    {
        vector<int32_t> colliding;
        for(int32_t i = 0; i != SizeL(agents); ++i)
        {
            auto& agent = *agents[i];
            if(agent.testCollisionAt(checkContext, agent.getPosition()))
            {
                colliding.push_back(i);
            }
        }
        if(colliding.empty())
        {
            return;
        }
        for(auto i:colliding)
        {
            paths[i] = nullptr;
            agents[i]->moveTo(startPositions[i]);
        }
    }
}

static void
CheckAndCollisionsAndRevert_Propagated(
        const vector<unique_ptr<iAgent>>& agents,
        vector<unique_ptr<iPath>>& paths,
        const vector<cPosition>& startPositions,
        const iCollisionContext* checkContext
        )
{
    vector<int32_t> colliding;
    vector<bool> flags(agents.size(), false);

    // only one full collision testing pass through agents
    for(int32_t i = 0; i != SizeL(agents); ++i)
    {
        auto& agent = *agents[i];
        assertD(static_cast<int32_t>(agent.getUserData()) == i);
        if(agent.testCollisionAt(checkContext, agent.getPosition()))
        {
            colliding.push_back(i);
            flags[i] = true;
        }
    }

    // and then collisions are propagated to other agents as necessary
    // with getAllAgentsOverlapped()
    while(!colliding.empty())
    {
        for(auto i:colliding)
        {
            paths[i] = nullptr;
            agents[i]->moveTo(startPositions[i]);
        }
        vector<unique_ptr<iAgent>> overlapped;
        vector<int32_t> propagatedColliding;
        for(auto i:colliding)
        {
            agents[i]->getAllAgentsOverlapped(checkContext, overlapped);
            for(const auto& agent:overlapped)
            {
                int32_t i = static_cast<int32_t>(agent->getUserData());
                if(!flags[i]) // only add once, in case overlapped by multiple
                {
                    flags[i] = true;
                    propagatedColliding.push_back(i);
                }
            }
        }
        propagatedColliding.swap(colliding);
    }
}

static void
AdvanceAgents_PointCollideAndRevert(
        const vector<unique_ptr<iAgent>>& agents, vector<unique_ptr<iPath>>& paths,
        const iCollisionContext* checkContext
        )
{
    assertD(agents.size() == paths.size());

    // store start positions

    vector<cPosition> startPositions;
    startPositions.reserve(agents.size());
    for(int32_t i = 0; i != SizeL(agents); ++i)
    {
        startPositions.push_back(agents[i]->getPosition());
    }

    // advance along paths without collision checking

    AdvanceAgents(agents, paths, nullptr);

    // then check collisions and revert as necessary

    //CheckAndCollisionsAndRevert_Simple(agents, paths, startPositions, checkContext);
    CheckAndCollisionsAndRevert_Propagated(agents, paths, startPositions, checkContext);
}

void
RunDemoWithMesh(iPathEngine* pathengine, iTestBed* testBed, iMesh& mesh)
{
    // create shapes
    unique_ptr<iShape> agentShape;
    {
        int32_t array[] =
        {
            -10,-10,
            -10,10,
            10,10,
            10,-10,
        };
        agentShape = pathengine->newShape(array, sizeof(array) / sizeof(*array));
    }

    {
        const char* preprocessAttributes[] =
        {
            "splitWithCircumferenceBelow",
            "5000",
            "smallConvex_WrapNonConvex",
            "true",
            nullptr
        };
        mesh.generateUnobstructedSpaceFor(*agentShape, true, preprocessAttributes);
    }
    mesh.generatePathfindPreprocessFor(*agentShape, nullptr);

    auto allContext = mesh.newContext();
    auto atRestContext = mesh.newContext();

    vector<unique_ptr<iAgent>> agents;
    for(int32_t i = 0; i != 2000; ++i)
    {
        cPosition p = RandomUnobstructedPosition(*agentShape, mesh, allContext.get());
        auto agent = mesh.placeAgent(*agentShape, p);
        agent->setUserData(i);
        allContext->addAgent(*agent);
        atRestContext->addAgent(*agent);
        agents.push_back(std::move(agent));
    }

    vector<unique_ptr<iPath>> paths(agents.size());
    vector<bool> agentInAtRestContext(agents.size(), true);

    cTimeTracker pathfindTracker;
    cTimeTracker advanceTracker;

    cMeshRenderGeometry meshRenderGeometry(*testBed, mesh);
    cAgentRenderGeometry agentRenderGeometry(*testBed, *agentShape, 40.f);

    cMeshLOSPreprocess losPreprocess(mesh);

    bool exitFlag = false;
    while(!exitFlag)
    {
        meshRenderGeometry.render(*testBed);

        clock_t start, finish;

        // update the at rest collision context
        for(int32_t i = 0; i != SizeL(agents); ++i)
        {
            bool isAtRest = !static_cast<bool>(paths[i]);
            if(isAtRest == agentInAtRestContext[i])
            {
                // no change required for this agent
                continue;
            }
            auto& agent = *agents[i];
            if(isAtRest)
            {
                atRestContext->addAgent(agent);
            }
            else
            {
                atRestContext->removeAgent(agent);
            }
            agentInAtRestContext[i] = isAtRest;
        }

        start = clock();
        FindPaths(agents, paths, atRestContext.get(), 30);
        finish = clock();
        pathfindTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);

        start = clock();
        //AdvanceAgents(agents, paths, allContext);
        AdvanceAgents_PointCollideAndRevert(agents, paths, allContext.get());
        finish = clock();
        advanceTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);

        // draw agent if it has been placed
        for(auto& agentPtr : agents)
        {
            testBed->setColour("red");
            agentRenderGeometry.renderAt(*testBed, mesh, agentPtr->getPosition());
        }

        testBed->setColour("purple");
        if(pathfindTracker.ready())
        {
            std::ostringstream oss;
            oss << "pathfinding (average per frame): " << pathfindTracker.get() << "s";
            testBed->printTextLine(0, oss.str().c_str());
        }
        if(advanceTracker.ready())
        {
            std::ostringstream oss;
            oss << "agent advance (average per frame): " << advanceTracker.get() << "s";
            testBed->printTextLine(0, oss.str().c_str());
        }

        CameraControl(*testBed, losPreprocess);
        testBed->update(20, exitFlag);

        // receive and process messages for all keys pressed since last frame
        const char *keypressed;
        while(keypressed = testBed->receiveKeyMessage())
        {
            if(keypressed[0] != 'd') // is it a key down message?
                continue;

            switch(keypressed[1])
            {
            case '_':
            {
                if(strcmp("ESCAPE", keypressed + 2) == 0)
                    exitFlag = true;
                break;
            }
            }
        }
    }
}

void
TestbedApplicationMain(iPathEngine *pathEngine, iTestBed *testBed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    unique_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/thainesford.tok", buffer);
        mesh = pathEngine->loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
    }
    ZoomExtents(*testBed, *mesh);
    RunDemoWithMesh(pathEngine, testBed, *mesh);
    assertR(!mesh->hasRefs());
}

