#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "project/testbedApp/MeshFederation/BuildFederation_FromWorldMesh.h"
#include "project/testbedApp/MeshFederation/BuildFederation_TileByTile.h"
#include "project/testbedApp/MeshFederation/BuildFederation_TileByTile3DCP.h"
#include "sampleShared/Error.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "common/STL_Helper.h"
#include "common/FileOutputStream.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include "base/Container/Vector.h"
#include "base/types/Error.h"
#include <sstream>
#include <math.h>
#include <string.h>
#include <memory>

using PE::vector;
using std::unique_ptr;

iTestBed* gTestBed;

void
ControlCamera(float& heading, float& elevation)
{
    if(!gTestBed->getRightMouseState())
    {
        return;
    }
    int32_t dx, dy;
    gTestBed->getMouseScreenDelta(dx, dy);
    heading += dx * 0.016f;
    heading = fmod(heading, 6.2856f);
    elevation += dy * 0.016f;
    if(elevation > 1.5f)
    {
        elevation = 1.5f;
    }
    else
    if(elevation < -0.05f)
    {
        elevation = -0.05f;
    }
}

static unique_ptr<iMesh>
LoadTileMesh(iTestBed* testBed, iPathEngine* pathEngine, int32_t tileIndex)
{
    std::ostringstream fileName;
    fileName << "federationTile" << tileIndex << ".tok";
    PE::vector<char> buffer;
    LoadBinary(fileName.str().c_str(), buffer);
    if(buffer.empty())
        return nullptr;
    return pathEngine->loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
}

static bool
GetUnobstructedPositionInTile(
        const iMeshFederation* federation,
        int32_t tileIndex,
        iMesh* tileMesh,
        iShape& agentShape,
        int32_t agentSize,
        cPosition& result
        )
{
    if(tileMesh == 0)
    {
        return false; // no geometry overlapped by this tile
    }
    result = tileMesh->generateRandomPosition();
    if(tileMesh->testPointCollision(agentShape, 0, result))
    {
        return false;
    }
    cHorizontalRange range;
    federation->getRepresentedRegion_Local(tileIndex, range);
    if(result.x < range.minX || result.x > range.maxX || result.y < range.minY || result.y > range.maxY)
    {
        return false; // position generated is outside the tiles represented region;
    }

    assertD(range.maxX - range.minX > agentSize * 2); // the tile overlap must be large than agent size
    assertD(range.maxY - range.minY > agentSize * 2);
    range.minX += agentSize;
    range.minY += agentSize;
    range.maxX -= agentSize;
    range.maxY -= agentSize;
    if(result.x < range.minX || result.x > range.maxX || result.y < range.minY || result.y > range.maxY)
    {
        return false; // agent would overlap outside federation world bounds at this position
    }

    return true;
}

static void
GetUnobstructedPositionInFederation(
        const iMeshFederation* federation,
        const vector<unique_ptr<iMesh>>& tileMeshes,
        iShape& agentShape,
        int32_t agentSize,
        int32_t& tileIndex, cPosition& positionInTile
        )
{
// loop through tiles, attempting to generate a position within each tile
// (in general, we can't guarantee to find a position in any given tile, because it's possible for individual tiles in a federation to be empty)
    while(1)
    {
        for(tileIndex = 0; tileIndex != federation->size(); ++tileIndex)
        {
            if(GetUnobstructedPositionInTile(
                    federation, tileIndex,
                    tileMeshes[tileIndex].get(),
                    agentShape, agentSize,
                    positionInTile
                    ))
            {
                return;
            }
        }
    }
}

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed *testBed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

// build a 'mesh federation' object,
// and an associated set of tiled mesh instances
// and save this all to disk

    // note that this mesh federation construction step would usually happen at content generation time
    // (and could be skipped after the first time the demo is run, as long as the world mesh is not changed)

    const int32_t tileStep = 8000;
    const int32_t overlap = 2000;
    const int32_t agentSize = 20;
    BuildFederation_FromWorldMesh(pathEngine, testBed, "thainesford_town", tileStep, overlap);

    // ** comment out the above and replace with the following to use 'tile by tile' federation construction
    // ** instead of construction from a supplied world mesh

    //const int32_t tileStep = 40000;
    //const int32_t overlap = 20000;
    //const int32_t agentSize = 160;
    //BuildFederation_TileByTile(pathEngine, testBed, "meshfederation_example", tileStep, overlap);

    // ** or use the following to construct from tiled 3D content processing result pieces
    // ** (after running the 3D content processing demo to generate a set of these result pieces)

    //const int32_t tileStep = 40000;
    //const int32_t overlap = 20000;
    //const int32_t agentSize = 160;
    //BuildFederation_TileByTile3DCP(pathEngine, testBed, tileStep, overlap);


// load the constructed mesh federation and associated tile meshes back from disk
// and run the demo against these loaded objects

    const int32_t agentHeight = agentSize * 6;
    const float agentSpeed = static_cast<float>(agentSize) * 0.75f;
    const float cameraDistance = static_cast<float>(agentSize) * 350.f;
    const int32_t resolveRange = agentSize * 3;

    gTestBed = testBed;

    unique_ptr<iMeshFederation> federation;
    {
        PE::vector<char> buffer;
        LoadBinary("federation.tok", buffer);
        federation = pathEngine->loadFederation("tok", VectorBuffer(buffer), SizeU(buffer));
    }

    unique_ptr<iShape> agentShape;
    {
        int32_t array[]=
        {
            -agentSize, agentSize,
            agentSize, agentSize,
            agentSize, -agentSize,
            -agentSize, -agentSize,
        };
        agentShape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

  // generate collision and pathfinding preprocess for the federation tiles
  // also prepare some per-tile collision contexts
  // display the meshes in the testbed while the preprocess is being generated

    // note that, in a real application,
    // the preprocess could be pregenerated at content time, and then loaded back in at runtime,
    // to avoid delays during application setup

    const char* attributes[3];
    attributes[0] = "splitWithCircumferenceBelow";
    attributes[1] = "2000";
    attributes[2] = 0;

    vector<unique_ptr<iMesh>> tileMeshes(federation->size());
    vector<cMeshRenderGeometry> tileMeshRenderGeometry;
    vector<cMeshLOSPreprocess> tileMeshLOSPreprocess;
    vector<unique_ptr<iRenderGeometry>> tileRangeBounds;
    vector<unique_ptr<iCollisionContext>> contexts(federation->size());
    for(int32_t i = 0; i != federation->size(); ++i)
    {
        tileMeshes[i] = LoadTileMesh(testBed, pathEngine, i);
        if(!tileMeshes[i])
        {
            continue;
        }

        tileMeshes[i]->generateUnobstructedSpaceFor(*agentShape, true, attributes);
        tileMeshes[i]->generatePathfindPreprocessFor(*agentShape, 0);

        {
            // set up a collision context for each tile
            // with query bounds that restrict pathfinding to the tile's represented region
            // query bounds act on agent origin, so the region is brought in by agentSize
            // so that the agent's shape does not overlap outside the represented region
            contexts[i] = tileMeshes[i]->newContext();
            cHorizontalRange rr;
            federation->getRepresentedRegion_Local(i, rr);
            assertD(rr.maxX - rr.minX > agentSize * 2); // the tile overlap must be large than agent size
            assertD(rr.maxY - rr.minY > agentSize * 2);
            rr.minX += agentSize;
            rr.minY += agentSize;
            rr.maxX -= agentSize;
            rr.maxY -= agentSize;
            contexts[i]->setQueryBounds(rr);
        }

        tileMeshRenderGeometry.emplace_back(*testBed, *tileMeshes[i]);

        tileMeshLOSPreprocess.emplace_back(*tileMeshes[i]);

        cHorizontalRange bounds;
        federation->getRepresentedRegion_Local(i, bounds);
        tileRangeBounds.push_back(testBed->newRenderGeometry());
        cMeshRenderGeometry::AddRangeBounds(*tileMeshes[i], bounds, *tileRangeBounds.back());
    }

    unique_ptr<iAgent> agent;
    int32_t agentTile = -1;

  // get an arbitrary, unobstructed position within the federation at which to start the agent
    {
        cPosition agentStartPosition;
        GetUnobstructedPositionInFederation(
                federation.get(), tileMeshes,
                *agentShape, agentSize,
                agentTile, agentStartPosition
                );
        agent = tileMeshes[agentTile]->placeAgent(*agentShape, agentStartPosition);
    }

    float cameraHeading = 0.f;
    float cameraElevation = 1.f;

    float agentHeading = 0;
    float agentPrecisionX = 0.f;
    float agentPrecisionY = 0.f;
    unique_ptr<iPath> path;
    bool markerSet = false;
    cPosition markerPosition;
    const char* markerColour;

    cAgentRenderGeometry agentRenderGeometry(*testBed, *agentShape, agentHeight);

    bool exitFlag = false;
    while(!exitFlag)
    {
        {
            cPosition pos = agent->getPosition();
            testBed->lookAt(
                static_cast<float>(pos.x) + agentPrecisionX,
                static_cast<float>(pos.y) + agentPrecisionY,
                static_cast<float>(tileMeshes[agentTile]->heightAtPosition(pos)),
                cameraHeading, cameraElevation,
                cameraDistance
                );
        }

        tileMeshRenderGeometry[agentTile].render(*testBed);

        testBed->setColour("red");
        testBed->render(*tileRangeBounds[agentTile]);

        testBed->setColour("orange");
        agentRenderGeometry.renderAt(*testBed, *tileMeshes[agentTile], agent->getPosition(), agentPrecisionX, agentPrecisionY);
        testBed->setColour("purple");
        cAgentRenderGeometry::DrawAgentHeading(*testBed, *agent, agentHeight, agentSize * 2, agentHeading);

      // draw path if there is one
        testBed->setColour("green");
        if(path)
            cMeshRenderGeometry::DrawPath(*testBed, *path);

      // draw marker, if set
        if(markerSet)
        {
            testBed->setColour(markerColour);
            cMeshRenderGeometry::Draw3DCross(*testBed, *tileMeshes[agentTile], markerPosition, agentSize * 2);
        }

        {
            bool windowClosed;
            testBed->update(40, windowClosed);
            if(windowClosed)
                exit(0);
        }

        bool repathRequested = false;

    // receive and process messages for all keys pressed since last frame
        const char* keyPressed;
        while(keyPressed = testBed->receiveKeyMessage())
        {
            if(keyPressed[0] != 'd' || keyPressed[1] != '_')
            {
              // only interested in key down messages for keys with extended key strings, here
                continue;
            }
            if(strcmp("ESCAPE", keyPressed + 2) == 0)
            {
                exitFlag = true;
            }
            else if(strcmp("LMOUSE", keyPressed + 2) == 0)
            {
                repathRequested = true;
            }
        }

        if(repathRequested || (path && path->size() == 1))
        {
          // clear the existing path

            path = nullptr;
            markerSet = false;

          // and check whether we need to translate to a neighbouring tile

            int32_t worldX, worldY;
            federation->getTileCentre(agentTile, worldX, worldY);
            worldX += agent->getPosition().x;
            worldY += agent->getPosition().y;
            int32_t tileForQuery = federation->tileForQueryCentre(worldX, worldY);
            if(tileForQuery != agentTile)
            {
                cPosition positionOnOverlapped = federation->translatePosition(
                        agentTile, *tileMeshes[agentTile],
                        tileForQuery, *tileMeshes[tileForQuery],
                        agent->getPosition()
                        );
                assertD(positionOnOverlapped.cell != -1);
                agentTile = tileForQuery;
                agent = tileMeshes[agentTile]->placeAgent(*agentShape, positionOnOverlapped);
            }
        }

        if(path)
        {
            if(path->size() >= 2)
            {
            // set heading from the vector for the current path section
                cPosition nextTarget = path->position(1);
                cPosition current = agent->getPosition();
                int32_t dx,dy;
                dx = nextTarget.x - current.x;
                dy = nextTarget.y - current.y;
                agentHeading = static_cast<float>(atan2(static_cast<double>(dx), static_cast<double>(dy)));
            }
            agent->advanceAlongPathWithPrecision(path.get(), agentSpeed, 0, agentPrecisionX, agentPrecisionY);
        }

        if(repathRequested)
        {
          // send agent to the position currently under the mouse cursor
          // note that we guaranteed, above, that the agent is in the handled region for agentTile

            cPosition target = tileMeshLOSPreprocess[agentTile].positionAtMouse(*testBed);

            if(target.cell != -1)
            {
                markerSet = true;
                markerPosition = target;
                cPosition unobstructed = tileMeshes[agentTile]->findClosestUnobstructedPosition(*agentShape, 0, target, resolveRange);
                if(unobstructed.cell == -1)
                {
                  // unable to find an unobstructed position within range
                    markerColour = "red";
                }
                else
                {
                    markerColour = "yellow";
                    path = agent->findShortestPathTo(contexts[agentTile].get(), unobstructed);
                }
            }
        }
    }
}
