
#include "base/types/Header.h"
#include "project/testbedApp/MeshFederation/BuildFederation_TileByTile.h"
#include "sampleShared/SimpleDOM.h"
#include "sampleShared/LoadWhiteSpaceDelimited.h"
#include "sampleShared/LoadContentChunkPlacement.h"
#include "common/FileOutputStream.h"
#include "externalAPI/i_pathengine.h"
#include <string>
#include <fstream>
#include <sstream>

using PE::vector;
using std::unique_ptr;

static void
GetWorldRange(
        const vector<unique_ptr<iContentChunkInstance>>& placedInstances,
        cHorizontalRange& result
        )
{
    assertD(!placedInstances.empty());
    placedInstances[0]->getHorizontalRange(result);
    int32_t i;
    for(i = 1; i != static_cast<int32_t>(placedInstances.size()); ++i)
    {
        cHorizontalRange instanceRange;
        placedInstances[i]->getHorizontalRange(instanceRange);
        if(instanceRange.minX < result.minX)
        {
            result.minX = instanceRange.minX;
        }
        else if(instanceRange.maxX > result.maxX)
        {
            result.maxX = instanceRange.maxX;
        }
        if(instanceRange.minY < result.minY)
        {
            result.minY = instanceRange.minY;
        }
        else if(instanceRange.maxY > result.maxY)
        {
            result.maxY = instanceRange.maxY;
        }
    }
}

static bool
RangesOverlap(const cHorizontalRange& r1, const cHorizontalRange& r2)
{
    if(r1.minX > r2.maxX)
    {
        return false;
    }
    if(r1.maxX < r2.minX)
    {
        return false;
    }
    if(r1.minY > r2.maxY)
    {
        return false;
    }
    if(r1.maxY < r2.minY)
    {
        return false;
    }
    return true;
}

static void
BuildTile(
        iTestBed* testBed,
        iMeshFederation& federation,
        int32_t tileIndex,
        const vector<unique_ptr<iContentChunkInstance>>& placedInstances
        )
{
    vector<const iContentChunkInstance*> overlappingInstances;

    cHorizontalRange representedRegion;
    federation.getRepresentedRegion_World(tileIndex, representedRegion);

    for(int32_t i = 0; i != static_cast<int32_t>(placedInstances.size()); ++i)
    {
        cHorizontalRange instanceRange;
        placedInstances[i]->getHorizontalRange(instanceRange);
        if(RangesOverlap(instanceRange, representedRegion))
        {
            overlappingInstances.push_back(placedInstances[i].get());
        }
    }

    const iContentChunkInstance** instancesBuffer = 0;
    if(!overlappingInstances.empty())
    {
        instancesBuffer = &overlappingInstances[0];
    }

    iOutputStream* snapshotOSPtr = 0;

    // the following lines can be uncommented to write a file containing a snapshot of the data being passed into content processing
    // this can be imported into 3DS Max, and is useful for troubleshooting any problems with the content processing
//    std::ostringstream snapshotFileName;
//    snapshotFileName << "../resource/meshes/federationTile" << tileIndex << ".sourceContent.tok";
//    cFileOutputStream snapshotFOS(snapshotFileName.str().c_str());
//    snapshotOSPtr = &snapshotFOS;

    auto tileMesh = federation.buildTileMeshFromContent(
            tileIndex,
            instancesBuffer, static_cast<int32_t>(overlappingInstances.size()),
            0, // no additional options
            *snapshotOSPtr
            );

    std::ostringstream fileName;
    fileName << "federationTile" << tileIndex << ".tok";
    cFileOutputStream fos(fileName.str().c_str());
    tileMesh->saveGround("tok", true, fos);
    testBed->printTextLine(5, fileName.str().c_str());
    testBed->printTextLine(5, "finished generating federation tile:");
    testBed->printTextLine(5, "Generating Federation Tile Meshes");
    bool windowClosed;
    testBed->update(20, windowClosed);
    if(windowClosed)
        exit(0);
}

void
BuildFederation_TileByTile(
        iPathEngine* pathEngine, iTestBed* testBed,
        const char* placementFileName,
        int32_t tileSize, int32_t overlap
        )
{
    cSimpleDOM placementScript;
    {
        std::string fileName("../resource/contentChunkPlacement/");
        fileName.append(placementFileName);
        fileName.append(".txt");
        std::ifstream is(fileName.c_str());
        if(!is.good())
        {
            Error("Fatal", "Could not open content chunk placement script.");
            return;
        }
        LoadWhiteSpaceDelimited(is, placementScript);
    }

    vector<unique_ptr<iContentChunkInstance>> placedInstances;
    LoadContentChunkPlacement(pathEngine, testBed, placementScript, placedInstances, true);
    if(placedInstances.empty())
    {
        Error("Fatal", "No content chunk instances placed.");
        return;
    }

    cHorizontalRange worldRange;
    GetWorldRange(placedInstances, worldRange);

    auto federation = pathEngine->buildMeshFederation_TilingOnly(worldRange, tileSize, overlap);

    for(int32_t i = 0; i != federation->size(); ++i)
    {
        BuildTile(testBed, *federation, i, placedInstances);
    }

    std::ostringstream fileName;
    fileName << "federation.tok";
    cFileOutputStream fos(fileName.str().c_str());
    federation->save("tok", fos);
}

