#include "base/types/Header.h"
#include "project/testbedApp/PlayableDemo/GameObject.h"
#include "project/testbedApp/PlayableDemo/GameState.h"
#include "project/testbedApp/PlayableDemo/Globals.h"
#include "project/testbedApp/PlayableDemo/iBehaviour.h"
#include "project/testbedApp/PlayableDemo/Resources.h"
#include "sampleShared/MoveAgent.h"
#include "sampleShared/SimpleDOM.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "externalAPI/i_pathengine.h"
#include <math.h>
#include <stdlib.h>

using std::string;
using std::unique_ptr;

bool
cGameObject::stepCurrentBehaviour()
{
    assertD(_currentBehaviour);
    bool completed = _currentBehaviour->update(*this);
    if(completed)
    {
        delete _currentBehaviour;
        _currentBehaviour = 0;
    }
    return completed;
}

void
cGameObject::setContentName(const std::string& contentName)
{
    _contentName = contentName;
    string::size_type pos = contentName.find(".");
    if(pos == string::npos)
    {
        _type = contentName;
    }
    else
    {
        _type = std::string(contentName, 0, pos);
        _name = std::string(contentName.c_str() + pos + 1);
    }
}

void
cGameObject::constructorCommon()
{
    _agent->setUserData(reinterpret_cast<int64_t>(this));

    _colour = _initTemplate->getAttribute("colour");
    _alpha = 1.0f;

    string attributeString;
    if(_initTemplate->hasAttribute("alpha"))
    {
        attributeString = _initTemplate->getAttribute("alpha");
        _alpha = static_cast<float>(strtod(attributeString.c_str(), 0));
    }
    attributeString = _initTemplate->getAttribute("height");
    _height = strtol(attributeString.c_str(), 0, 10);

    _toggledColour = _colour;
    _toggledAlpha = _alpha;
    _toggledHeight = _height;

    int32_t i = _initTemplate->firstChildWithName("toggled");
    if(i >= 0)
    {
        if(_initTemplate->_children[i].hasAttribute("colour"))
        {
            _toggledColour = _initTemplate->_children[i].getAttribute("colour");
        }
        if(_initTemplate->_children[i].hasAttribute("alpha"))
        {
            attributeString = _initTemplate->_children[i].getAttribute("alpha");
            _toggledAlpha = static_cast<float>(strtod(attributeString.c_str(), 0));
        }
        if(_initTemplate->_children[i].hasAttribute("height"))
        {
            attributeString = _initTemplate->_children[i].getAttribute("height");
            _toggledHeight = strtol(attributeString.c_str(), 0, 10);
        }
    }

    _headingIndicatorSize = _initTemplate->attributeAsLongWithDefault("headingIndicatorSize", 0);
    if(_headingIndicatorSize)
    {
        _headingIndicatorColour = _initTemplate->getAttribute("headingIndicatorColour");
    }

    _speed = _initTemplate->attributeAsFloatWithDefault("speed", 0.f);

    _isPushable = (_initTemplate->getAttribute("isPushable") == "true");

    _precisionX = 0.0f;
    _precisionY = 0.0f;

    _toggled = false;
    _visible = true;
    if(_initTemplate->getAttribute("startsInvisible") == "true")
    {
        _visible = false;
    }

    _currentBehaviour = 0;
    _currentTarget = 0;

    _renderGeometry = gTestBed->newRenderGeometry();
    if(_agent->isMoveable())
    {
        _shape = _agent->getShape();
        cAgentRenderGeometry::AddCylinder(*_shape, static_cast<float>(_height), *_renderGeometry);
    }
    else
        cAgentRenderGeometry::AddCylinder(*_agent, static_cast<float>(_height), *_renderGeometry);
    if(_toggledHeight == _height)
        _toggledRenderGeometry = _renderGeometry->addExternalRef();
    else
    {
        _toggledRenderGeometry = gTestBed->newRenderGeometry();
        if(_agent->isMoveable())
            cAgentRenderGeometry::AddCylinder(*_shape, static_cast<float>(_toggledHeight), *_toggledRenderGeometry);
        else
            cAgentRenderGeometry::AddCylinder(*_agent, static_cast<float>(_toggledHeight), *_toggledRenderGeometry);
    }
}

cGameObject::cGameObject(int32_t index,
                         iMesh& mesh, const cPosition& position,
                        const cSimpleDOM& templates, const std::string& contentName, float orientation)
{
    _index = index;
    setContentName(contentName);
    int32_t i = templates.firstChildWithName(_type);
    assertR(i >= 0);
    _initTemplate = &templates._children[i];

    _mesh = mesh.addExternalRef();

    {
        int32_t shapeIndex = _initTemplate->attributeAsLong("shape");
        auto shape = gResources->refShape(shapeIndex).addExternalRef();
        _agent = mesh.placeAgent(*shape, position);
    }

    _heading = orientation;

    constructorCommon();
}

cGameObject::cGameObject(int32_t index, iMesh& mesh, const cSimpleDOM& templates, const std::string& contentName, unique_ptr<iAgent> agent) :
 _agent(std::move(agent))
{
    _index = index;
    setContentName(contentName);
    int32_t i = templates.firstChildWithName(_type);
    assertR(i >= 0);
    _initTemplate = &templates._children[i];

    _mesh = mesh.addExternalRef();

    _heading = 0.f;

    constructorCommon();
}

void
cGameObject::completeInitialisation()
{
    _collisionFlags.resize(gGameState->numberOfObjects());
    _collisionFlagsValid = false;
    setDefaultBehaviour();
    setDefaultTarget();
}

cGameObject::~cGameObject()
{
    delete _currentBehaviour;
}


unique_ptr<iPath>
cGameObject::findShortestPathTo(const cPosition& target) const
{
    return _agent->findShortestPathTo(&gGameState->refObstructionsContext(), target);
}

unique_ptr<iPath>
cGameObject::findShortestPathTo_IgnoringOneObject(const cPosition& target, cGameObject& toIgnore) const
{
    unique_ptr<iPath> result;
    if(gGameState->refObstructionsContext().includes(*toIgnore._agent))
    {
        gGameState->refObstructionsContext().removeAgent(*toIgnore._agent);
        result = _agent->findShortestPathTo(&gGameState->refObstructionsContext(), target);
        gGameState->refObstructionsContext().addAgent(*toIgnore._agent);
    }
    else
    {
        result = _agent->findShortestPathTo(&gGameState->refObstructionsContext(), target);
    }
    return result;
}

unique_ptr<iPath>
cGameObject::findPathAway(const cPosition& awayFrom, int32_t distanceAway) const
{
    return _agent->findPathAway(&gGameState->refObstructionsContext(), awayFrom, distanceAway);
}

unique_ptr<iPath>
cGameObject::findStraightLinePathTo(const cPosition& target) const
{
    if(_agent->testCollisionTo(&gGameState->refObstructionsContext(), target))
    {
        return nullptr;
    }
    auto result = _agent->findShortestPathTo(&gGameState->refObstructionsContext(), target);
    assertD(result);
    return result;
}

void
cGameObject::moveUnderDirectControl(float localX, float localY, float speedMultiplier)
{
    localX *= _speed * speedMultiplier;
    localY *= _speed * speedMultiplier;
    double sinOf = sin(double(_heading));
    double cosOf = cos(double(_heading));
    double dx = _precisionX + sinOf * localY + cosOf * localX;
    double dy = _precisionY + cosOf * localY - sinOf * localX;
    int32_t dxL = static_cast<int32_t>(dx);
    int32_t dyL = static_cast<int32_t>(dy);

    if(dxL == 0 && dyL == 0)
    {
    // no movement after approximation
        _precisionX = static_cast<float>(dx);
        _precisionY = static_cast<float>(dy);
        return;
    }

    cPosition current = _agent->getPosition();
    cPosition target;
    target.x = current.x + dxL;
    target.y = current.y + dyL;

    cCollidingLine collidingLine;
    {
        unique_ptr<iAgent> collidingAgent;
        bool collides = _agent->firstCollisionTo(&gGameState->refObstructionsContext(), target.x, target.y, target.cell, collidingLine, collidingAgent);
        if(!collides)
        {
            _precisionX = static_cast<float>(dx - dxL);
            _precisionY = static_cast<float>(dy - dyL);
            _agent->moveTo(target);
            return;
        }

        if(collidingAgent)
        {
            cGameObject* collidingGameObject = reinterpret_cast<cGameObject*>(collidingAgent->getUserData());
            if(collidingGameObject->_isPushable)
            {
                MoveAgent(gPathEngine, &gGameState->refObstructionsContext(), &gGameState->refCollisionCheckersContext(),
                          collidingGameObject->_agent.get(), _heading, localX, localY,
                          _precisionX, _precisionY
                          );
            }
        }
    }

    MoveAgent(gPathEngine, &gGameState->refObstructionsContext(),
                      _agent.get(), _heading, localX, localY,
                      _precisionX, _precisionY
                      );
}

bool
cGameObject::moveInDirection(float dX, float dY)
{
    float offsetX = _precisionX + dX;
    float offsetY = _precisionY + dY;
    cPosition currentP = getPosition();
    cPosition targetP;
    targetP.x = static_cast<int32_t>(currentP.x + offsetX);
    targetP.y = static_cast<int32_t>(currentP.y + offsetY);
    bool collided = _agent->testCollisionTo_XY(&gGameState->refObstructionsContext(), targetP.x, targetP.y, targetP.cell);
    if(!collided)
    {
        _agent->moveTo(targetP);
        int32_t offsetXL = targetP.x - currentP.x;
        int32_t offsetYL = targetP.y - currentP.y;
        _precisionX = offsetX - offsetXL;
        _precisionY = offsetY - offsetYL;
    }
    return collided;
}

bool
cGameObject::advanceAlongPath(iPath* path, float speedMultiplier)
{
    if(path->size() >= 2)
    {
        cPosition nextTarget = path->position(1);
        cPosition current = path->position(0);
        int32_t dx, dy;
        dx = nextTarget.x - current.x;
        dy = nextTarget.y - current.y;
        _heading = static_cast<float>(atan2(static_cast<double>(dx), static_cast<double>(dy)));
    }
    return _agent->advanceAlongPathWithPrecision(path, _speed * speedMultiplier, &gGameState->refObstructionsContext(), _precisionX, _precisionY);
}

void
cGameObject::faceTowards(const cPosition& target)
{
    cPosition p = getPosition();
    int32_t dx, dy;
    dx = target.x - p.x;
    dy = target.y - p.y;
    _heading = static_cast<float>(atan2(static_cast<double>(dx), static_cast<double>(dy)));
}

float
cGameObject::distanceTo(const cGameObject& rhs) const
{
    cPosition p1 = _agent->getPosition();
    cPosition p2 = rhs._agent->getPosition();
    float dx = static_cast<float>(p2.x - p1.x);
    float dy = static_cast<float>(p2.y - p1.y);
    return static_cast<float>(sqrt(dx * dx + dy * dy));
}
float
cGameObject::distanceTo(const cPosition& p) const
{
    cPosition p1 = _agent->getPosition();
    float dx = static_cast<float>(p.x - p1.x);
    float dy = static_cast<float>(p.y - p1.y);
    return static_cast<float>(sqrt(dx * dx + dy * dy));
}

void
cGameObject::setDefaultTarget()
{
    std::string targetName = _initTemplate->getAttribute("defaultTarget");
    if(!targetName.empty())
    {
        _currentTarget = gGameState->findObject(targetName);
    }
}

void
cGameObject::setBehaviour(const cSimpleDOM& element)
{
    if(_currentBehaviour)
    {
        delete _currentBehaviour;
    }
    _currentBehaviour = iBehaviour::create(element, *this);
}
void
cGameObject::setDefaultBehaviour()
{
    if(_currentBehaviour)
    {
        delete _currentBehaviour;
    }
    int32_t i = _initTemplate->firstChildWithName("defaultBehaviour");
    if(i >= 0)
    {
        _currentBehaviour = iBehaviour::create(_initTemplate->_children[i], *this);
    }
    else
    {
        _currentBehaviour = 0;
    }
}

void
cGameObject::invalidateCollisionFlags()
{
    _collisionFlagsValid = false;
}
bool
cGameObject::collisionFlagsAreValid() const
{
    return _collisionFlagsValid;
}
void
cGameObject::initialiseCollisionFlags()
{
    std::fill(_collisionFlags.begin(), _collisionFlags.end(), false);
    _collisionFlagsValid = true;
}
void
cGameObject::setCollisionFlag(cGameObject& collisionTarget)
{
    _collisionFlags[collisionTarget._index] = true;
}
bool
cGameObject::testCollisionFlag(cGameObject& collisionTarget)
{
    return _collisionFlags[collisionTarget._index] != 0;
}

void
cGameObject::update()
{
    if(_currentBehaviour)
    {
        gCurrentObject = this;
        stepCurrentBehaviour();
        gCurrentObject = 0;
    }
}

void
cGameObject::draw(bool inAdditiveBlendingPhase)
{
    if(!_visible)
    {
        return;
    }
    int32_t usedHeight;
    float alpha = _toggled ? _toggledAlpha : _alpha;
    bool useAdditiveBlending = (alpha < 1.0f);
    if(useAdditiveBlending != inAdditiveBlendingPhase)
    {
        return;
    }
    if(useAdditiveBlending)
    {
        gTestBed->setAdditiveBlendingAlpha(alpha);
    }

    gTestBed->setColour(_toggled ? _toggledColour.c_str() : _colour.c_str());
    usedHeight = _toggled ? _toggledHeight : _height;
    iRenderGeometry& rg = (_toggled ? *_toggledRenderGeometry : *_renderGeometry);
    if(_shape)
    {
        cPosition pos = _agent->getPosition();
        gTestBed->render_Offset(
            rg,
            static_cast<float>(pos.x),
            static_cast<float>(pos.y),
            _mesh->heightAtPositionF(pos)
            );
    }
    else
    {
        gTestBed->render(rg);
    }
    if(_headingIndicatorSize)
    {
        gTestBed->setColour(_headingIndicatorColour.c_str());
        cAgentRenderGeometry::DrawAgentHeading(*gTestBed, *_agent, static_cast<float>(usedHeight), static_cast<float>(_headingIndicatorSize), _heading);
    }
    if(_currentBehaviour)
    {
        _currentBehaviour->draw(*this);
    }
}
