
#include "externalAPI/i_pathengine.h"
#include "base/Container/Vector.h"
#include <string>
#include <memory>

class cSimpleDOM;
class iBehaviour;
class cGameState;

class cGameObject
{
  // initialised by constructor
    std::string _contentName;
    std::string _type;
    std::string _name;
    const cSimpleDOM* _initTemplate;
    uint32_t _index;
    std::unique_ptr<iMesh> _mesh;
    std::unique_ptr<const iShape> _shape;
    std::unique_ptr<iAgent> _agent;
    std::unique_ptr<iRenderGeometry> _renderGeometry;
    std::unique_ptr<iRenderGeometry> _toggledRenderGeometry;
    float _precisionX, _precisionY;
    bool _isPushable;
    float _speed;
    int32_t _height;
    int32_t _toggledHeight;

  // initialised in completeInitalisation()
    bool _collisionFlagsValid;
    PE::vector<char> _collisionFlags;
    iBehaviour* _currentBehaviour;
    cGameObject* _currentTarget;

    // returns true if the current behaviour completed
    bool stepCurrentBehaviour();

    void setContentName(const std::string& contentName);
    void constructorCommon();

public:

  // purely graphical attributes
  // these can be accessed and changed at will, for example by script commands
    std::string _colour;
    float _alpha;
    std::string _toggledColour;
    float _toggledAlpha;
    std::string _headingIndicatorColour;
    int32_t _headingIndicatorSize;
    bool _visible; // note that visibility and toggled status are used to control script flow
    bool _toggled;
    float _heading;

// at initial construction time, objects are placed and any properties not dependant on other objects or the game state are set

  // shapes index is supplied by template, placed at an anchor
    cGameObject(int32_t index, iMesh& mesh, const cPosition& position, const cSimpleDOM& templates, const std::string& contentName, float orientation);
  // obstruction info was retrieved directly from the mesh as an agent
    cGameObject(int32_t index, iMesh& mesh, const cSimpleDOM& templates, const std::string& contentName, std::unique_ptr<iAgent> agent);

// this method is then called once all objects have been placed and the game state has been initialised
// at this point the default behaviour is created, default target set, etc.
    void completeInitialisation();

    ~cGameObject();

// accessor methods
    const std::string& contentName() const {return _contentName;}
    const std::string& name() const {return _name;}
    const std::string& type() const {return _type;}
    const iShape& refShape() const {return *_shape;}
    iAgent& refAgent() const {return *_agent;}
    cPosition getPosition() {return _agent->getPosition();}
    float getPrecisionX() const {return _precisionX;}
    float getPrecisionY() const {return _precisionY;}
    float getSpeed() const {return _speed;}
    cGameObject* getTarget() const {return _currentTarget;}
    const cSimpleDOM& refTemplate() const {return *_initTemplate;}

// call through to PathEngine
//  the obstructions context is applied
//  in the case of failed path, unset pointer is returned
    std::unique_ptr<iPath> findShortestPathTo(const cPosition& target) const;
    std::unique_ptr<iPath> findShortestPathTo_IgnoringOneObject(const cPosition& target, cGameObject& toIgnore) const;
    std::unique_ptr<iPath> findPathAway(const cPosition& awayFrom, int32_t distanceAway) const;
    std::unique_ptr<iPath> findStraightLinePathTo(const cPosition& target) const;

// agent movement
    void moveUnderDirectControl(float localX, float localY, float speedMultiplier);
    bool moveInDirection(float dX, float dY);
    bool advanceAlongPath(iPath* path, float speedMultiplier = 1.f);

    void faceTowards(const cPosition& target);

    float distanceTo(const cGameObject& rhs) const;
    float distanceTo(const cPosition& p) const;

    void setDefaultTarget();
    void setTarget(cGameObject* value) {_currentTarget = value;}
    void setSpeed(float value) {_speed = value;}

    void setBehaviour(const cSimpleDOM& element);
    void setDefaultBehaviour();

// these collision flags are used to speed up collision checking against objects that are designated as switches
    void invalidateCollisionFlags();
    bool collisionFlagsAreValid() const;
    void initialiseCollisionFlags();
    void setCollisionFlag(cGameObject& collisionTarget);
    bool testCollisionFlag(cGameObject& collisionTarget);

    void update();
    void draw(bool inAdditiveBlendingPhase);
};
