#include "base/types/Header.h"
#include "project/testbedApp/PlayableDemo/PackHunterBehaviour.h"
#include "project/testbedApp/PlayableDemo/Globals.h"
#include "project/testbedApp/PlayableDemo/GameState.h"
#include "project/testbedApp/PlayableDemo/GameObject.h"
#include "project/testbedApp/PlayableDemo/PathFrequency.h"
#include "sampleShared/MoveAgent.h"
#include "sampleShared/SimpleDOM.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "externalAPI/i_pathengine.h"
#include <stdlib.h>

using std::string;

cQueueManager cPackHunterBehaviour::_queueManager;
int32_t cPackHunterBehaviour::_timeSincePathToGoal = 0;
PE::vector<cPosition> cPackHunterBehaviour::_interceptPositions;


cPackHunterBehaviour::cPackHunterBehaviour(const cSimpleDOM& element, cGameObject& controlledObject) :
    _controlledObject(controlledObject),
    _queueEntry(_queueManager, element.attributeAsLong("queueCollisionShape"), controlledObject.refAgent().addExternalRef())
{
    cGameObject* target = controlledObject.getTarget();
    assertD(target);
    _interceptPositions.push_back(target->getPosition());

    _debugDisplay = element.attributeAsBoolWithDefault("debugDisplay", false);

    _currentPath = 0;
    _timeSincePathQuery = 0;
}

cPackHunterBehaviour::~cPackHunterBehaviour()
{
    _interceptPositions.pop_back();
}

bool
cPackHunterBehaviour::update(cGameObject& controlledObject)
{
    cGameObject* target = controlledObject.getTarget();
    if(!target)
    {
        return true; // failed
    }

// update intercept positions if these are too old

    if(_queueEntry.isLeader())
        // so that timer will update roughly once per frame
        // also this entry's distance to target is used below
    {
        _timeSincePathToGoal++;
        if(_timeSincePathToGoal >= 6)
        {
            cGameObject* goal = gGameState->findObject("goal");
            std::unique_ptr<iPath> pathToGoal = target->findShortestPathTo(goal->getPosition());
            if(pathToGoal)
            {
                float interceptDistance = _queueEntry.getDistanceToTarget() * 0.6f;
                if(interceptDistance > 700.f)
                {
                    interceptDistance = 700.f;
                }
                const iShape& targetShape = target->refShape();
                for(int32_t i = 0; i < static_cast<int32_t>(_interceptPositions.size()); ++i)
                {
                    float precisionX, precisionY;
                    pathToGoal->advanceAlong(targetShape, interceptDistance, &gGameState->refObstructionsContext(), precisionX, precisionY);
                    _interceptPositions[i] = pathToGoal->position(0);
                }
            }
            else
            {
                int32_t i;
                for(i = 0; i < static_cast<int32_t>(_interceptPositions.size()); ++i)
                {
                    _interceptPositions[i] = target->getPosition();
                }
            }
            _timeSincePathToGoal = 0;
        }
    }

// update position in pack order
    _queueEntry.updateDistanceToTarget(controlledObject.distanceTo(*target));

    _timeSincePathQuery++;

    float speedMultiplier = 1.f;

    if(controlledObject.distanceTo(*target) < 320.f)
    {
        std::unique_ptr<iPath> straightLinePath = controlledObject.findStraightLinePathTo(target->getPosition());
        if(straightLinePath)
        {
            _timeSincePathQuery = 0;
            _currentPath = std::move(straightLinePath);
            speedMultiplier = 1.4f;
        }
    }

    if(_timeSincePathToGoal > 0 && _timeSincePathQuery >= 5
     && (_currentPath == 0 || _currentPath->position(_currentPath->size() - 1) != _interceptPositions[_queueEntry.queueIndex()])
     )
    {
        _timeSincePathQuery = 0;
        _currentPath = controlledObject.findShortestPathTo(_interceptPositions[_queueEntry.queueIndex()]);
    }

    if(_currentPath)
    {
        if(_queueEntry.testQueueCollision())
        {
            speedMultiplier = 0.7f;
        }
        bool blocked = controlledObject.advanceAlongPath(_currentPath.get(), speedMultiplier);
        if(blocked)
        {
            _currentPath.reset();
        }
    }
    else
    {
        controlledObject.faceTowards(target->getPosition());
    }
    return false;
}

void
cPackHunterBehaviour::draw(cGameObject& controlledObject)
{
    if(!_debugDisplay)
    {
        return;
    }
    if(_currentPath)
    {
        if(_queueEntry.isLeader())
        {
            gTestBed->setColour("orange");
        }
        else
        {
            gTestBed->setColour("green");
        }
        cMeshRenderGeometry::DrawPath(*gTestBed, *_currentPath);
    }
}
