//**********************************************************************
//
// Copyright (c) 2007
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "project/testbedApp/SemiDynamicObstacles/DoubleBufferedObstacleSet.h"
#include "project/testbedApp/SemiDynamicObstacles/DynamicAgentManager.h"
#include "sampleShared/PartitionedTerrain.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/CameraControl.h"
#include "sampleShared/Error.h"
#include "base/Container/Vector.h"
#include "base/types/Error.h"
#include "common/STL_Helper.h"
#include "common/FileOutputStream.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <string>
#include <string.h>
#include <sstream>
#include <math.h>
#include <time.h>
#include <memory>

using std::unique_ptr;
using PE::vector;

class cAverageTimeTracker
{
    vector<double> _times;
    int32_t _bufferPosition;
    bool _bufferFull;
public:
    cAverageTimeTracker(int32_t bufferSize) :
      _times(bufferSize),
      _bufferPosition(0),
      _bufferFull(false)
    {
        assertD(bufferSize > 0);
    }
    void addTiming(double value)
    {
        _times[_bufferPosition++] = value;
        if(_bufferPosition == SizeL(_times))
        {
            _bufferPosition = 0;
            _bufferFull = true;
        }
    }
    bool ready() const
    {
        return _bufferFull;
    }
    double get()
    {
        assertD(ready());
        double result = _times[0];
        int32_t i;
        for(i = 1; i != SizeL(_times); ++i)
        {
            result += _times[i];
        }
        return result /= static_cast<double>(SizeL(_times));
    }
};

static void
RotateShapeCoords(int32_t vertices, const double* vertexCoords, double rotateBy, PE::vector<int32_t>& result)
{
    double sinOf = sin(rotateBy);
    double cosOf = cos(rotateBy);
    result.resize(vertices * 2);
    int32_t i;
    for(i = 0; i != vertices; ++i)
    {
        double x = vertexCoords[i * 2];
        double y = vertexCoords[i * 2 + 1];
        double rotatedX = sinOf * y + cosOf * x;
        double rotatedY = cosOf * y - sinOf * x;
        result[i * 2] = static_cast<int32_t>(rotatedX);
        result[i * 2 + 1] = static_cast<int32_t>(rotatedY);
    }
}

static void
PreGenerateRotationShapes(
        iPathEngine* pathEngine,
        int32_t vertices, const double* vertexCoords,
        vector<unique_ptr<iShape>>& result
        )
{
    static const int32_t NUMBER_OF_ROTATIONS = 30;
    static const double PI = 3.14159265358979323;
    result.reserve(NUMBER_OF_ROTATIONS);
    vector<int32_t> rotatedCoords;
    for(int32_t i = 0; i != NUMBER_OF_ROTATIONS; ++i)
    {
        double theta = PI * 2 * i / NUMBER_OF_ROTATIONS;
        RotateShapeCoords(vertices, vertexCoords, theta, rotatedCoords);
        if(!pathEngine->shapeIsValid(&rotatedCoords[0], vertices))
        {
            Error("Fatal", "Shape coordinates are not valid after rotation. Avoid small edges and angles near 180.");
        }
        result.push_back(std::move(pathEngine->newShape(&rotatedCoords[0], vertices)));
    }
}

static unique_ptr<iAgent>
RandomlyPlaceObstacle(iMesh& mesh, const cPosition& centre, int32_t range, const vector<unique_ptr<iShape>>& shapeRotations)
{
    int i = rand() % SizeL(shapeRotations);
    cPosition p = mesh.generateRandomPositionLocally(centre, range);
    return mesh.placeAgent(*shapeRotations[i], p);
}

static unique_ptr<iAgent>
PlaceWithRandomRotation(
        iMesh& mesh, const vector<unique_ptr<iShape>>& shapeRotations, const cPosition& p
        )
{
    int i = rand() % SizeL(shapeRotations);
    return mesh.placeAgent(*shapeRotations[i], p);
}

static void
RandomlyPlaceObstacles(
        iMesh& mesh,
        const cPosition& centre,
        const vector<unique_ptr<iShape>>& shapeRotations,
        int32_t numberToAdd,
        int32_t range,
        cDoubleBufferedObstacleSet& addTo
        )
{
    int32_t i;
    for(i = 0; i != numberToAdd; ++i)
    {
        unique_ptr<iAgent> placed(RandomlyPlaceObstacle(mesh, centre, range, shapeRotations));
        addTo.addObstacle(*placed);
    }
}

static void
GenerateWallShapeCoords(
        int32_t startX, int32_t startY,
        int32_t endX, int32_t endY,
        int32_t radius,
        vector<int32_t>& result
        )
{
    if(startX > endX || (startX == endX && startY > endY))
    {
        std::swap(startX, endX);
        std::swap(startY, endY);
    }
    if(startX == endX || startY == endY)
    {
        result.push_back(startX - radius);
        result.push_back(startY - radius);
        result.push_back(startX - radius);
        result.push_back(endY + radius);
        result.push_back(endX + radius);
        result.push_back(endY + radius);
        result.push_back(endX + radius);
        result.push_back(startY - radius);
        return;
    }
    if(startY > endY)
    {
        result.push_back(startX - radius);
        result.push_back(startY - radius);
        result.push_back(startX - radius);
        result.push_back(startY + radius);
        result.push_back(startX + radius);
        result.push_back(startY + radius);
        result.push_back(endX + radius);
        result.push_back(endY + radius);
        result.push_back(endX + radius);
        result.push_back(endY - radius);
        result.push_back(endX - radius);
        result.push_back(endY - radius);
        return;
    }
    result.push_back(startX + radius);
    result.push_back(startY - radius);
    result.push_back(startX - radius);
    result.push_back(startY - radius);
    result.push_back(startX - radius);
    result.push_back(startY + radius);
    result.push_back(endX - radius);
    result.push_back(endY + radius);
    result.push_back(endX + radius);
    result.push_back(endY + radius);
    result.push_back(endX + radius);
    result.push_back(endY - radius);
}

static unique_ptr<iAgent>
PlaceWallAgent(
        const cPosition& start,
        const cPosition& end,
        int32_t radius,
        iMesh& mesh
        )
{
    assertD(radius >= 1);
    PE::vector<int32_t> coords;
    GenerateWallShapeCoords(start.x, start.y, end.x, end.y, radius, coords);
    return mesh.placeLargeStaticObstacle(&coords[0], static_cast<uint32_t>(coords.size()), start);
}

static unique_ptr<iMesh>
BuildBaseGround(iPathEngine* pathEngine, iTestBed* testBed)
{
  // generate terrain stand-in geometry for the range -10000,-10000 -> 10000,10000
  // world coordinates are in centimetres, so this corresponds to a 200 metres by 200 metres region centred on the origin
    cPartitionedTerrain terrain(-10000, -10000, 2000, 10);
    //cPartitionedTerrain terrain(-30000, -30000, 2000, 30);

    PE::vector<const iFaceVertexMesh*> groundParts;
    groundParts.push_back(&terrain);

    const char* options[5];
    options[0] = "partitionTranslationTo3D";
    options[1] = "true";
    options[2] = "useIdentityMapping";
    options[3] = "true";
    options[4] = 0;
    return pathEngine->buildMeshFromContent(&groundParts.front(), SizeL(groundParts), options);
}

static void
DemoMain(iPathEngine* pathEngine, iTestBed* testBed, iMesh& mesh)
{
    unique_ptr<iShape> agentShape;
    {
        int32_t array[]=
        {
            -20,-20,
            -20,20,
            20,20,
            20,-20,
        };
        agentShape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    unique_ptr<iShape> erasorShape;
    {
        int32_t array[]=
        {
            -2000,-2000,
            -2000,2000,
            2000,2000,
            2000,-2000,
        };
        erasorShape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    vector<unique_ptr<iShape>> rockShapes;
    {
        double array[]=
        {
            -180., 60.,
            -60., 100.,
            210., 20.,
            186., -36.,
            120., -48.,
            -180., -60.,
        };
        PreGenerateRotationShapes(pathEngine, sizeof(array) / sizeof(*array), array, rockShapes);
    }
    vector<unique_ptr<iShape>> hutShapes;
    {
        double array[]=
        {
            -250., -200.,
            -250., 200.,
            250., 200.,
            250., -200.
        };
        PreGenerateRotationShapes(pathEngine, sizeof(array) / sizeof(*array), array, hutShapes);
    }

    testBed->setColour("white");
    testBed->printTextLine(10, "Generating base mesh preprocess...");
    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    mesh.generateUnobstructedSpaceFor(*erasorShape, false, 0);
    // pre-combining base mesh preprocess can help speed up obstacle set preprocess update
    // in cases where there is base mesh obstruction detail
    // (although this is not actually the case here)
    mesh.generateUnobstructedSpaceFor(*agentShape, true, 0);

    testBed->setColour("white");
    testBed->printTextLine(10, "Initialising double buffered obstacle set...");
    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    const char* preprocessAttributes[] =
    {
        "splitWithCircumferenceBelow",
        "5000",
        "smallConvex_WrapNonConvex",
        "true",
        0
    };
    cDoubleBufferedObstacleSet set(testBed, mesh, *agentShape, preprocessAttributes);

    testBed->setColour("white");
    testBed->printTextLine(10, "Initialising double buffered obstacle set... (completed)");
    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    ZoomExtents(*testBed, mesh); // place camera to see all of newly loaded mesh

    testBed->setColour("white");
    testBed->printTextLine(10, "Initialising dynamic agents...");
    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    set.lockForeground();
    cDynamicAgentManager dynamicAgents(testBed, &mesh, agentShape.get(), set.refForegroundCollisionContext(), 1000);
    set.unlockForeground();

    testBed->setColour("white");
    testBed->printTextLine(10, "Initialising dynamic agents... (completed)");
    {
        bool windowClosed;
        testBed->update(20, windowClosed);
        if(windowClosed)
            exit(0);
    }

    bool erasorOn = false;
    bool displayTimings = false;

    cAverageTimeTracker moveAlongTimeTracker(40);
    cAverageTimeTracker rePathTimeTracker(40);
    cAverageTimeTracker drawObstaclesTimeTracker(40);
    cAverageTimeTracker drawAgentsTimeTracker(40);
    cAverageTimeTracker assignRegionsTimeTracker(40);

    cPosition wallStart, wallEnd;

    cMeshRenderGeometry meshRenderGeometry(*testBed, mesh, *agentShape);
    cAgentRenderGeometry erasorRenderGeometry(*testBed, *erasorShape, 20.f);

    cMeshLOSPreprocess losPreprocess(mesh);

    while(1)
    {
    // generate or regenerate set of placed obstacle
        bool regenerate = false;
        do
        {
            {
                set.lockForeground();
                assertR(!set.refForegroundPreprocessedSet().pathfindPreprocessNeedsUpdate(*agentShape));
                clock_t start, finish;
                start = clock();
                dynamicAgents.moveAlongPaths(set.refForegroundCollisionContext());
                finish = clock();
                moveAlongTimeTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);
                start = clock();
                dynamicAgents.rePath(set.refForegroundPreprocessedSet(), set.refForegroundCollisionContext(), 20);
                finish = clock();
                rePathTimeTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);
                set.unlockForeground();
            }
            {
                clock_t start, finish;
                start = clock();
                set.lockForeground();
                dynamicAgents.assignRegions(set.refForegroundPreprocessedSet());
                set.unlockForeground();
                finish = clock();
                assignRegionsTimeTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);
            }

        // draw mesh and mesh static geometry elements
            meshRenderGeometry.render(*testBed);

        // draw obstacles in double buffered set
            {
                clock_t start, finish;
                set.lockForeground();
                start = clock();
                testBed->setColour("white");
                set.renderPreprocessed();
                testBed->setColour("orange");
                set.renderDynamic();
                testBed->setColour("red");
                set.renderToBeDeleted();
                testBed->setColour("orange");
                set.renderPreprocessedExpansion();
                finish = clock();
                drawObstaclesTimeTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);
                set.unlockForeground();
            }

            {
                clock_t start, finish;
                start = clock();
                dynamicAgents.renderAgents_ColouredByRegion();
                finish = clock();
                drawAgentsTimeTracker.addTiming(static_cast<double>(finish - start) / CLOCKS_PER_SEC);
            }

            testBed->setColour("orange");
            {
                std::ostringstream oss;
                set.lockForeground();
                oss << set.numberOfPreprocessedObstacles() << " preprocessed obstacles, ";
                oss << set.numberOfDynamicObstacles() << " dynamic obstacles, ";
                oss << set.refForegroundPreprocessedSet().getNumberOfConnectedRegions(*agentShape) << " connected regions";
                oss << ", " << dynamicAgents.size()  << " moving agents";
                set.unlockForeground();
                testBed->printTextLine(0, oss.str().c_str());
            }
            if(set.updateInProgress())
            {
                testBed->printTextLine(0, "Preprocess update in progress..");
            }
            if(displayTimings)
            {
                testBed->setColour("purple");
                if(!set.updateInProgress())
                {
                    std::ostringstream oss;
                    oss << "last preprocess update time: " << set.getLastUpdateTime();
                    testBed->printTextLine(0, oss.str().c_str());
                    //{
                    //    std::ostringstream oss;
                    //    oss << "last buffer swap time: " << set.getLastBufferSwapTime();
                    //    testBed->printTextLine(10, oss.str().c_str());
                    //}
                }
                if(moveAlongTimeTracker.ready())
                {
                    std::ostringstream oss;
                    oss << "advance along paths (average per frame): " << moveAlongTimeTracker.get() << "s";
                    testBed->printTextLine(0, oss.str().c_str());
                }
                if(rePathTimeTracker.ready())
                {
                    std::ostringstream oss;
                    oss << "new path generation (average per frame): " << rePathTimeTracker.get() << "s";
                    testBed->printTextLine(0, oss.str().c_str());
                }
                if(drawObstaclesTimeTracker.ready())
                {
                    std::ostringstream oss;
                    oss << "draw obstacles (average per frame): " << drawObstaclesTimeTracker.get() << "s";
                    testBed->printTextLine(0, oss.str().c_str());
                }
                if(drawAgentsTimeTracker.ready())
                {
                    std::ostringstream oss;
                    oss << "draw agents (average per frame): " << drawAgentsTimeTracker.get() << "s";
                    testBed->printTextLine(0, oss.str().c_str());
                }
                if(assignRegionsTimeTracker.ready())
                {
                    std::ostringstream oss;
                    oss << "assign regions (average per frame): " << assignRegionsTimeTracker.get() << "s";
                    testBed->printTextLine(0, oss.str().c_str());
                }
            }
            testBed->setColour("lightgrey");
            testBed->printTextLine(0, "Press 'T' to toggle timing info");

        // render anything that needs to be displayed with additive blending

            testBed->enterAdditiveBlendingPhase();

            if(wallStart.cell != -1)
            {
                testBed->setColour("green");
                unique_ptr<iAgent> a;
                if(testBed->getKeyState("_SHIFT"))
                {
                    a = PlaceWallAgent(wallStart, wallEnd, 150, mesh);
                }
                else
                {
                    a = PlaceWallAgent(wallStart, wallEnd, 100, mesh);
                }
                testBed->setAdditiveBlendingAlpha(0.5f);
                std::unique_ptr<iRenderGeometry> renderGeometry = testBed->newRenderGeometry();
                cAgentRenderGeometry::AddCylinder(*a, 20, *renderGeometry);
                testBed->render(*renderGeometry);
            }

            if(erasorOn)
            {
                cPosition p = losPreprocess.positionAtMouse(*testBed);
                if(p.cell != -1)
                {
                    testBed->setColour("red");
                    testBed->setAdditiveBlendingAlpha(0.5f);
                    erasorRenderGeometry.renderAt(*testBed, mesh, p);
                }
            }

            CameraControl(*testBed, losPreprocess);

        // tell the testBed to render this frame
            {
                bool windowClosed;
                testBed->update(20, windowClosed);
                if(windowClosed)
                    exit(0);
            }

        // receive and process messages for all keys pressed since last frame
            cPosition p = losPreprocess.positionAtMouse(*testBed);
            const char* keyPressed;
            while(keyPressed = testBed->receiveKeyMessage())
            {
                if(keyPressed[0] != 'd') // is it a key down message?
                    continue;

                switch(keyPressed[1])
                {
                case '_':
                    {
                        if(!strcmp("ESCAPE", keyPressed + 2))
                        {
                            return;
                        }
                        if(!strcmp("SPACE", keyPressed + 2) && !set.updateInProgress())
                        {
                            regenerate = true;
                        }
                        break;
                    }
                case 'H':
                    if(p.cell != -1)
                    {
                        if(testBed->getKeyState("_SHIFT"))
                        {
                            RandomlyPlaceObstacles(mesh, p, hutShapes, 30, 2000, set);
                        }
                        else
                        {
                            unique_ptr<iAgent> agent = PlaceWithRandomRotation(mesh, hutShapes, p);
                            set.addObstacle(*agent);
                        }
                    }
                    break;
                case 'R':
                    if(p.cell != -1)
                    {
                        if(testBed->getKeyState("_SHIFT"))
                        {
                            RandomlyPlaceObstacles(mesh, p, rockShapes, 25, 1500, set);
                        }
                        else
                        {
                            unique_ptr<iAgent> agent = PlaceWithRandomRotation(mesh, rockShapes, p);
                            set.addObstacle(*agent);
                        }
                    }
                    break;
                case 'W':
                    if(wallStart.cell != -1)
                    {
                        unique_ptr<iAgent> agent;
                        if(testBed->getKeyState("_SHIFT"))
                        {
                            agent = PlaceWallAgent(wallStart, wallEnd, 150, mesh);
                        }
                        else
                        {
                            agent = PlaceWallAgent(wallStart, wallEnd, 100, mesh);
                        }
                        set.addObstacle(*agent);
                        wallStart = wallEnd;
                    }
                    break;
                case 'D':
                    if(p.cell != -1)
                    {
                        set.lockForeground();
                        vector<unique_ptr<iAgent>> overlapped;
                        mesh.getAllAgentsOverlapped(
                                erasorOn ? *erasorShape : *agentShape,
                                &set.refForegroundCollisionContext(),
                                p,
                                overlapped
                                );
                        set.unlockForeground();
                        for(int32_t i = 0; i != overlapped.size(); ++i)
                        {
                            set.removeObstacle(*overlapped[i]);
                        }
                    }
                    break;
                case 'E':
                    erasorOn = !erasorOn;
                    break;
                case 'T':
                    displayTimings = !displayTimings;
                    break;
                case 'C':
                    if(p.cell != -1)
                    {
                        if(testBed->getKeyState("_SHIFT") || wallStart.cell == -1)
                        {
                          // start new wall
                            wallStart = p;
                        }
                        wallEnd = p;
                    }
                    break;
                case 'S':
                    {
                        set.storeToNamedObstacles();
                        cFileOutputStream fos("../resource/meshes/semiDynamicObstacles.tok");
                        mesh.saveGround("tok", true, fos);
                    }
                    break;
                }
            }
        }
        while(regenerate == false);
        set.startUpdate();
    }
}

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed* testBed)
{
    // check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion() != TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion() < TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal", "Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion() != PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion() < PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal", "Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    // load an existing mesh, if previously saved out
    // or if the file is not present then build a new mesh
    unique_ptr<iMesh> mesh;
    if(FileExists("../resource/meshes/semiDynamicObstacles.tok"))
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/semiDynamicObstacles.tok", buffer);
        mesh = pathEngine->loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
    }
    else
    {
        mesh = BuildBaseGround(pathEngine, testBed);
    }

    DemoMain(pathEngine, testBed, *mesh);

    assertD(!mesh->hasRefs());
}