//**********************************************************************
//
// Copyright (c) 2007-2013
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#include "base/types/Header.h"
#include "project/testbedApp/SemiDynamicObstacles/DoubleBufferedObstacleSet.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "base/MutexLock.h"
#include "externalAPI/i_pathengine.h"
#include "platform_common/Thread.h"
#include <algorithm>
#include <sstream>
#include <memory>
#include <string.h>
#include <time.h>

using std::unique_ptr;

namespace
{

class cThread : public iThread
{
public:
    cDoubleBufferedObstacleSet* _set;
    cThread()
    {
        _set = 0;
    }
    void
    run()
    {
        assertR(_set);
        _set->doUpdate();
    }
};

} // anonymous namespace

cThread gThread;

void
cDoubleBufferedObstacleSet::doUpdate()
{
    assertD(updateInProgress());

    {
        clock_t start, finish;
        start = clock();
        _background->updatePathfindPreprocessFor(*_agentShape);
        finish = clock();
        _lastUpdateTime = static_cast<double>(finish - start) / CLOCKS_PER_SEC;
    }

    clock_t start, finish;
    start = clock();
    {
        cMutexLock lock(_bufferSwapMutex);
        for(int32_t i = 0; i != SizeL(_beingRemoved); ++i)
        {
            _foreground->removeAgent(*_beingRemoved[i]);
            assertD(!_background->includes(*_beingRemoved[i]));
        }
        _beingRemoved.resize(0);

        _foregroundContext->removeObstacleSet(*_foreground);
        _foregroundContext->removeObstacleSet(*_beingAddedOverlay);

        while(!_beingAddedOverlay->empty())
        {
            auto agent = _beingAddedOverlay->getAgent(0);
            _foreground->addAgent(*agent);
            _beingAddedOverlay->removeAgent(*agent);
        }

        std::swap(_foreground, _background);

        _foregroundContext->addObstacleSet(*_foreground);

        _renderPreprocessValid = false;
        _expansionsRenderPreprocessValid = false;
        _toBeDeletedRenderPreprocessValid = false;
    }
    finish = clock();
    _lastBufferSwapTime = static_cast<double>(finish - start) / CLOCKS_PER_SEC;

    cMutexLock lock(_updateInProgressMutex);
    _updateInProgress = false;
}

cDoubleBufferedObstacleSet::cDoubleBufferedObstacleSet(iTestBed* testBed, iMesh& mesh, iShape& agentShape, const char *const* preprocessAttributes)
{
    _testBed = testBed;
    _expansionsRenderPreprocess = 0;
    _expansionsRenderPreprocessValid = false;
    _renderPreprocessValid = false;
    _toBeDeletedRenderPreprocessValid = false;
    _lastUpdateTime = 0.0;
    _lastBufferSwapTime = 0.0;
    _mesh = mesh.addExternalRef();
    _agentShape = agentShape.addExternalRef();
    PE::vector<const char*> combinedAttributes;
    if(preprocessAttributes)
    {
        while(preprocessAttributes[0])
        {
            assertD(preprocessAttributes[1]);
            const char* attribute(preprocessAttributes[0]);
            const char* value(preprocessAttributes[1]);
            preprocessAttributes += 2;
            if(strcmp("markForPreprocessing", attribute) == 0)
            {
                Error("NonFatal", "'markForPreprocessing' attribute should not be supplied for cDoubleBufferedObstacleSet construction, over-ridden.");
                continue;
            }
            combinedAttributes.push_back(attribute);
            combinedAttributes.push_back(value);
        }
    }
    combinedAttributes.push_back(0);
    _toAddNextUpdateOverlay = _mesh->newObstacleSet_WithAttributes(&combinedAttributes[0]);
    combinedAttributes.pop_back();
    combinedAttributes.push_back(0);
    _beingAddedOverlay = _mesh->newObstacleSet_WithAttributes(&combinedAttributes[0]);
    combinedAttributes.pop_back();
    combinedAttributes.push_back("markForPreprocessing");
    combinedAttributes.push_back("true");
    combinedAttributes.push_back(0);
    _foreground = _mesh->newObstacleSet_WithAttributes(&combinedAttributes[0]);
    _background = _mesh->newObstacleSet_WithAttributes(&combinedAttributes[0]);
    _foregroundContext = _mesh->newContext();
    _foregroundContext->addObstacleSet(*_foreground);
    _foregroundContext->addObstacleSet(*_toAddNextUpdateOverlay); // this obstacle set is permanently included in the context, for simplicity
    _updateInProgress = false;
    for(int32_t i = 0; i != _mesh->getNumberOfNamedObstacles(); ++i)
    {
        unique_ptr<iAgent> agent;
        const char* id_ignored;
        _mesh->retrieveNamedObstacleByIndex(i, agent, id_ignored);
        _foreground->addAgent(*agent);
        _background->addAgent(*agent);
    }
    _foreground->updatePathfindPreprocessFor(*_agentShape);
    _foregroundLocked = false;
}
cDoubleBufferedObstacleSet::~cDoubleBufferedObstacleSet()
{
    assertD(!updateInProgress());
    assertD(!_foregroundLocked);
}

void
cDoubleBufferedObstacleSet::storeToNamedObstacles() const
{
    cMutexLock lock(_bufferSwapMutex);
    _mesh->clearAllNamedObstacles();
    for(int32_t i = 0; i != _foreground->size(); ++i)
    {
        std::ostringstream oss;
        oss << i;
        auto agent = _foreground->getAgent(i);
        _mesh->storeNamedObstacle(oss.str().c_str(), *agent);
    }
}

void
cDoubleBufferedObstacleSet::addObstacle(iAgent& toAdd)
{
    cMutexLock lock(_bufferSwapMutex);
    _toAddNextUpdateOverlay->addAgent(toAdd);
}
void
cDoubleBufferedObstacleSet::removeObstacle(iAgent& toRemove)
{
    cMutexLock lock(_bufferSwapMutex);
    if(_toAddNextUpdateOverlay->includes(toRemove))
    {
        _toAddNextUpdateOverlay->removeAgent(toRemove);
    }
    else
    {
        auto wrapped = toRemove.addExternalRef();
        if(std::find(_toRemoveNextUpdate.begin(), _toRemoveNextUpdate.end(), wrapped) == _toRemoveNextUpdate.end()) // check not already flagged for removal
        {
            _toRemoveNextUpdate.push_back(std::move(wrapped));
        }
        _toBeDeletedRenderPreprocessValid = false;
    }
}

void
cDoubleBufferedObstacleSet::startUpdate()
{
    assertD(!updateInProgress());
    _updateInProgress = true;
    assertD(_beingAddedOverlay->empty());
    for(int32_t i = 0; i != _toAddNextUpdateOverlay->size(); ++i)
    {
        unique_ptr<iAgent> agent(_toAddNextUpdateOverlay->getAgent(i));
        _beingAddedOverlay->addAgent(*agent);
        _background->addAgent(*agent);
    }
    _toAddNextUpdateOverlay->clear();
    for(int32_t i = 0; i != SizeL(_toRemoveNextUpdate); ++i)
    {
        _background->removeAgent(*_toRemoveNextUpdate[i]);
        _beingRemoved.push_back(std::move(_toRemoveNextUpdate[i]));
    }
    _toRemoveNextUpdate.clear();
    _foregroundContext->addObstacleSet(*_beingAddedOverlay);
    gThread._set = this;
    StartThread(gThread);
}

bool
cDoubleBufferedObstacleSet::updateInProgress() const
{
    cMutexLock lock(_updateInProgressMutex);
    return _updateInProgress;
}

double
cDoubleBufferedObstacleSet::getLastUpdateTime() const
{
    assertD(!updateInProgress()); // could also return this during update, with mutex protection
    return _lastUpdateTime;
}
double
cDoubleBufferedObstacleSet::getLastBufferSwapTime() const
{
    assertD(!updateInProgress()); // could also return this during update, with mutex protection
    return _lastBufferSwapTime;
}

void
cDoubleBufferedObstacleSet::lockForeground()
{
    assertD(!_foregroundLocked);
    _foregroundLocked = true;
    _bufferSwapMutex.lock();
}
void
cDoubleBufferedObstacleSet::unlockForeground()
{
    assertD(_foregroundLocked);
    _foregroundLocked = false;
    _bufferSwapMutex.unlock();
}

const iCollisionContext&
cDoubleBufferedObstacleSet::refForegroundCollisionContext() const
{
    assertD(_foregroundLocked);
    return *_foregroundContext;
}

const iObstacleSet&
cDoubleBufferedObstacleSet::refForegroundPreprocessedSet() const
{
    assertD(_foregroundLocked);
    return *_foreground;
}

int32_t
cDoubleBufferedObstacleSet::numberOfPreprocessedObstacles() const
{
    assertD(_foregroundLocked);
    return _foreground->size();
}
int32_t
cDoubleBufferedObstacleSet::numberOfDynamicObstacles() const
{
    assertD(_foregroundLocked);
    return _beingAddedOverlay->size() + _toAddNextUpdateOverlay->size();
}

void
cDoubleBufferedObstacleSet::renderPreprocessed() const
{
    assertD(_foregroundLocked);
    if(!_renderPreprocessValid)
    {
        _renderPreprocess = _testBed->newRenderGeometry();
        for(int32_t i = 0; i != _foreground->getNumberOfAgents(); ++i)
        {
            cAgentRenderGeometry::AddCylinder(_foreground->refAgent(i), 40, *_renderPreprocess);
        }
        _renderPreprocessValid = true;
    }
    _testBed->render(*_renderPreprocess);
}
void
cDoubleBufferedObstacleSet::renderPreprocessedExpansion() const
{
    assertD(_foregroundLocked);
    if(!_expansionsRenderPreprocessValid)
    {
        _expansionsRenderPreprocess = _testBed->newRenderGeometry();
        for(int32_t i = 0; i != _foreground->getNumberOfAgents(); ++i)
        {
            cAgentRenderGeometry::AddAgentExpansion(_foreground->refAgent(i), *_agentShape, *_expansionsRenderPreprocess);
        }
        _expansionsRenderPreprocessValid = true;
    }
    _testBed->render(*_expansionsRenderPreprocess);
}
void
cDoubleBufferedObstacleSet::renderDynamic() const
{
    assertD(_foregroundLocked);
    auto renderGeometry = _testBed->newRenderGeometry();
    for(int32_t i = 0; i != _beingAddedOverlay->getNumberOfAgents(); ++i)
        cAgentRenderGeometry::AddCylinder(_beingAddedOverlay->refAgent(i), 40, *renderGeometry);
    for(int32_t i = 0; i != _toAddNextUpdateOverlay->getNumberOfAgents(); ++i)
        cAgentRenderGeometry::AddCylinder(_toAddNextUpdateOverlay->refAgent(i), 40, *renderGeometry);
    _testBed->render(*renderGeometry);
}
void
cDoubleBufferedObstacleSet::renderToBeDeleted() const
{
    assertD(_foregroundLocked);
    if(!_toBeDeletedRenderPreprocessValid)
    {
        _toBeDeletedRenderPreprocess = _testBed->newRenderGeometry();
        for(int32_t i = 0; i != SizeL(_beingRemoved); ++i)
        {
            cAgentRenderGeometry::AddCylinder(*_beingRemoved[i], 40, *_toBeDeletedRenderPreprocess);
        }
        for(int32_t i = 0; i != SizeL(_toRemoveNextUpdate); ++i)
        {
            cAgentRenderGeometry::AddCylinder(*_toRemoveNextUpdate[i], 40, *_toBeDeletedRenderPreprocess);
        }
        _toBeDeletedRenderPreprocessValid = true;
    }
    _testBed->render(*_toBeDeletedRenderPreprocess);
}
