//**********************************************************************
//
// Copyright (c) 2007-2013
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#pragma once

// Provides obstacle set double buffering for situations where semi-dynamic obstacles will be added and removed,
// but not moved or otherwise modified (e.g with changes in cost to traverse)
//
// In this case a single iAgent can be created for each obstacle and shared by foreground and background obstacle sets.

#include "base/Container/Vector.h"
#include "platform_common/Mutex.h"
#include <memory>

class iTestBed;
class iObstacleSet;
class iCollisionContext;
class iMesh;
class iShape;
class iAgent;
class iRenderGeometry;

class cDoubleBufferedObstacleSet
{
    iTestBed* _testBed;
    std::unique_ptr<iMesh> _mesh;
    std::unique_ptr<iShape> _agentShape;
    std::unique_ptr<iObstacleSet> _foreground;
    std::unique_ptr<iObstacleSet> _background;
    std::unique_ptr<iObstacleSet> _beingAddedOverlay;
    PE::vector<std::unique_ptr<iAgent>> _beingRemoved;
    PE::vector<std::unique_ptr<iAgent>> _toRemoveNextUpdate;
    std::unique_ptr<iObstacleSet> _toAddNextUpdateOverlay;
    std::unique_ptr<iCollisionContext> _foregroundContext;
    bool _foregroundLocked;
    mutable std::unique_ptr<iRenderGeometry> _expansionsRenderPreprocess;
    mutable bool _expansionsRenderPreprocessValid;
    mutable std::unique_ptr<iRenderGeometry> _renderPreprocess;
    mutable bool _renderPreprocessValid;
    mutable std::unique_ptr<iRenderGeometry> _toBeDeletedRenderPreprocess;
    mutable bool _toBeDeletedRenderPreprocessValid;
    mutable cMutex _bufferSwapMutex;

    bool _updateInProgress;
    mutable cMutex _updateInProgressMutex;
    double _lastUpdateTime;
    double _lastBufferSwapTime;

public:

    void doUpdate(); // to be run in worker thread, should not be called by external code

    // any named obstacle on the mesh will be added to the set on construction
    cDoubleBufferedObstacleSet(iTestBed* testBed, iMesh& mesh, iShape& agentShape, const char *const* preprocessAttributes);
    ~cDoubleBufferedObstacleSet();

    // replaces existing named obstacles
    void storeToNamedObstacles() const; // automatically locks the foreground internally

    // must be a newly created agent
    // added agents will then be managed internally by this object
    void addObstacle(iAgent& toAdd);  // automatically locks the foreground internally

    // if this agent is a dynamic agent that has not yet been included in an update then it will be removed straight away
    // otherwise removal will take effect on completion of next update
    void removeObstacle(iAgent& toRemove);  // automatically locks the foreground internally

    void startUpdate();

    bool updateInProgress() const;
    double getLastUpdateTime() const;
    double getLastBufferSwapTime() const;

    void lockForeground();
    void unlockForeground();

    // following to only be called whilst foreground is locked
    const iCollisionContext& refForegroundCollisionContext() const;
    const iObstacleSet& refForegroundPreprocessedSet() const;
    int32_t numberOfPreprocessedObstacles() const;
    int32_t numberOfDynamicObstacles() const;
    void renderPreprocessed() const;
    void renderPreprocessedExpansion() const;
    void renderDynamic() const;
    void renderToBeDeleted() const;
};
