//**********************************************************************
//
// Copyright (c) 2012
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#include "base/types/Header.h"
#include "project/testbedApp/SeparatingGroups/AgentAI.h"
#include "externalAPI/i_pathengine.h"
#include <stddef.h>

using PE::vector;
using std::unique_ptr;

class cMovementOrder
{
    PE::vector<cPosition> _waypoints;
    bool _repeat;
    int32_t _nextWaypoint;
    int32_t _numberInProgress;
public:
    cMovementOrder(
            const PE::vector<cPosition>& waypoints,
            bool repeat
            ) :
     _waypoints(waypoints)
    {
        assertD(!_waypoints.empty());
        _repeat = repeat;
        _numberInProgress = 0;
        _nextWaypoint = 0;
    }
    bool
    canProceed() const
    {
        return _numberInProgress == 0;
    }
    bool
    isComplete() const
    {
        assertD(canProceed());
        return _nextWaypoint == SizeL(_waypoints);
    }
    cPosition
    getNextTarget(int32_t numberOfAgents)
    {
        assertD(canProceed());
        assertD(!isComplete());
        cPosition result = _waypoints[_nextWaypoint];
        ++_nextWaypoint;
        if(_repeat && _nextWaypoint == SizeL(_waypoints))
        {
            _nextWaypoint = 0;
        }
        _numberInProgress = numberOfAgents;
        return result;
    }
    void
    notifyAgentCompletedOrAbandonedWaypoint()
    {
        assertD(_numberInProgress);
        _numberInProgress--;
    }
};

void
cAgentAI::pathsToSingleTarget(const PE::vector<int32_t>& selectedAgents, const cPosition& targetPosition)
{
    // abort current paths if any
    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        _agentPaths[i] = nullptr;
    }

    // remove all from at rest context
    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        if(_atRestContext->includes(*_agents[i]))
        {
            _atRestContext->removeAgent(*_agents[i]);
            _avoidContext->addAgent(*_avoidAgents[i]);
        }
    }

    // and then do pathfinding
    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        _agentPaths[i] = std::shared_ptr<iPath>(_agents[i]->findShortestPathTo(_atRestContext.get(), targetPosition));
        if(!_agentPaths[i])
        {
            _atRestContext->addAgent(*_agents[i]);
            _avoidContext->removeAgent(*_avoidAgents[i]);
        }
    }
}

cAgentAI::cAgentAI(std::shared_ptr<iMesh> mesh, std::shared_ptr<iShape> agentShape, std::shared_ptr<iShape> avoidShape) :
    _atRestContext(mesh->newContext()),
    _avoidContext(mesh->newContext())
{
    _mesh = mesh;
    _agentShape = agentShape;
    _avoidShape = avoidShape;

    cPosition placementCentre;
    do
    {
        placementCentre = mesh->generateRandomPosition();
    }
    while(_mesh->testPointCollision(*_agentShape, 0, placementCentre));
    for(int32_t i = 0; i != 20; ++i)
    {
        cPosition p;
        do
        {
            p = _mesh->generateRandomPositionLocally(placementCentre, 1000);
        }
        while(_mesh->testPointCollision(*_agentShape, _atRestContext.get(), p));
        _agents.push_back(std::shared_ptr<iAgent>(_mesh->placeAgent(*_agentShape, p)));
        _avoidAgents.push_back(std::shared_ptr<iAgent>(_mesh->placeAgent(*_avoidShape, p)));
        _agents.back()->setUserData(i);
        _avoidAgents.back()->setUserData(i);
        _agentPaths.push_back(nullptr);
        _atRestContext->addAgent(*_agents.back());
    }
    _agentMovementOrders.resize(_agents.size(), -1);
}

cAgentAI::~cAgentAI()
{
    for(int32_t i = 0; i != SizeL(_movementOrders); ++i)
    {
        delete _movementOrders[i];
    }
}

void
cAgentAI::stopOrder(const PE::vector<int32_t>& selectedAgents)
{
    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        _agentPaths[i] = nullptr;
        if(_agentMovementOrders[i] != -1)
        {
            _movementOrders[_agentMovementOrders[i]]->notifyAgentCompletedOrAbandonedWaypoint();
            _agentMovementOrders[i] = -1;
        }
        if(!_atRestContext->includes(*_agents[i]))
        {
            _atRestContext->addAgent(*_agents[i]);
            _avoidContext->removeAgent(*_avoidAgents[i]);
        }
    }
}
void
cAgentAI::moveOrder(const PE::vector<int32_t>& selectedAgents, const cPosition& targetPosition)
{
    pathsToSingleTarget(selectedAgents, targetPosition);

    // notify existing movement orders about abandon, if any
    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        if(_agentMovementOrders[i] != -1)
        {
            _movementOrders[_agentMovementOrders[i]]->notifyAgentCompletedOrAbandonedWaypoint();
            _agentMovementOrders[i] = -1;
        }
    }
}

void
cAgentAI::moveOrder(const PE::vector<int32_t>& selectedAgents, const PE::vector<cPosition>& waypoints, bool repeat)
{
    stopOrder(selectedAgents);

  // get a free slot for the new movement order

    int32_t slot;
    for(slot = 0; slot != SizeL(_movementOrders); ++slot)
    {
        if(!_movementOrders[slot])
        {
            break;
        }
    }
    if(slot == SizeL(_movementOrders))
    {
        _movementOrders.push_back(0);
    }

  // create the movement order

    _movementOrders[slot] = new cMovementOrder(waypoints, repeat);

  // assign to the selected agents

    for(int32_t j = 0; j != SizeL(selectedAgents); ++j)
    {
        int32_t i = selectedAgents[j];
        _agentMovementOrders[i] = slot;
    }
}

void
cAgentAI::update()
{
    const float agentSpeed = 16.f;
    const float agentSpeed_Queuing = 10.f;
    const int32_t positionResolveDistance = 1500;
    int32_t i;

    // update movement orders
    for(int32_t orderI = 0; orderI != SizeL(_movementOrders); ++orderI)
    {
        cMovementOrder* order = _movementOrders[orderI];
        if(order == 0 || !order->canProceed())
        {
            continue;
        }
        if(order->isComplete())
        {
            delete order;
            _movementOrders[orderI] = 0;
            for(i = 0; i != SizeL(_agents); ++i)
            {
                if(_agentMovementOrders[i] == orderI)
                {
                    assertD(_agentPaths[i] == 0);
                    _agentMovementOrders[i] = -1;
                }
            }
            continue;
        }
        PE::vector<int32_t> assignedAgents;
        for(i = 0; i != SizeL(_agents); ++i)
        {
            if(_agentMovementOrders[i] == orderI)
            {
                assignedAgents.push_back(i);
            }
        }
        if(assignedAgents.empty())
        {
            delete order;
            _movementOrders[orderI] = 0;
            continue;
        }
        cPosition target = order->getNextTarget(SizeL(assignedAgents));
        pathsToSingleTarget(assignedAgents, target);
        for(int32_t j = 0; j != SizeL(assignedAgents); ++j)
        {
            int32_t i = assignedAgents[j];
            if(!_agentPaths[i])
            {
                order->notifyAgentCompletedOrAbandonedWaypoint();
            }
        }
    }

    // advance agents
    PE::vector<float> speeds(_agents.size(), agentSpeed);
    {
        vector<unique_ptr<iAgent>> overlappedAgents;
        overlappedAgents.reserve(_agents.size());
        for(i = 0; i != SizeL(_agents); ++i)
        {
            if(_agentPaths[i] == 0)
            {
                continue;
            }
            uint32_t pathLength = _agentPaths[i]->getLength();
            overlappedAgents.clear();
            _mesh->getAllAgentsOverlapped(*_agentShape, _avoidContext.get(), _avoidAgents[i]->getPosition(), overlappedAgents);
            for(int32_t j = 0; j != overlappedAgents.size(); ++j)
            {
                if(overlappedAgents[j].get() == _agents[i].get())
                {
                    continue;
                }
                auto& overlappedAgent = *overlappedAgents[j];
                int64_t overlappedI = overlappedAgent.getUserData();
                //.... also check that the agents have same destination!
                if(_agentPaths[static_cast<uint32_t>(overlappedI)] != 0 && _agentPaths[static_cast<uint32_t>(overlappedI)]->getLength() < pathLength)
                {
                    speeds[i] = agentSpeed_Queuing;
                    break;
                }
            }
        }
    }
    for(i = 0; i != SizeL(_agents); ++i)
    {
        if(_agentPaths[i] == 0)
        {
            continue;
        }
        assertD(!_atRestContext->includes(*_agents[i]));
        bool collided = _agents[i]->advanceAlongPath(_agentPaths[i].get(), speeds[i], _atRestContext.get());
        _avoidAgents[i]->moveTo(_agents[i]->getPosition());
        if(collided || _agentPaths[i]->size() == 1)
        {
            _agentPaths[i] = nullptr;
            assertD(!_atRestContext->includes(*_agents[i]));
            if(_mesh->testPointCollision(*_agentShape, _atRestContext.get(), _agents[i]->getPosition()))
            {
                cPosition target = _mesh->findClosestUnobstructedPosition(*_agentShape, _atRestContext.get(), _agents[i]->getPosition(), positionResolveDistance);
                if(target.cell != -1)
                {
                    _agentPaths[i] = std::shared_ptr<iPath>(_agents[i]->findShortestPathTo(_atRestContext.get(), target));
                }
            }
            if(!_agentPaths[i])
            {
                _atRestContext->addAgent(*_agents[i]);
                _avoidContext->removeAgent(*_avoidAgents[i]);
                if(_agentMovementOrders[i] != -1)
                {
                    _movementOrders[_agentMovementOrders[i]]->notifyAgentCompletedOrAbandonedWaypoint();
                }
            }
        }
    }
}
