//**********************************************************************
//
// Copyright (c) 2012
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#include "base/types/Header.h"
#include "project/testbedApp/SeparatingGroups/SeparatingGroups.h"
#include "project/testbedApp/SeparatingGroups/AgentAI.h"
#include "sampleShared/SelectAndMove.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/CameraControl.h"
#include "common/STL_Helper.h"
#include "base/Geometry/HorizontalRange_Helper.h"
#include "base/Container/Vector.h"
#include "externalAPI/i_pathengine.h"
#include <sstream>
#include <memory>
#include <string.h>

using std::unique_ptr;

static unique_ptr<iShape>
SquareShape(iPathEngine& pathEngine, int32_t radius)
{
    int32_t array[]=
    {
        -radius,radius,
        radius,radius,
        radius,-radius,
        -radius,-radius,
    };
    return pathEngine.newShape(array, sizeof(array) / sizeof(*array));
}

void
SeparatingGroups(iTestBed& testbed, iPathEngine& pathEngine)
{
    const int32_t clickResolveDistance = 1500;

    std::shared_ptr<iShape> agentShape(SquareShape(pathEngine, 10));
    std::shared_ptr<iShape> avoidShape(SquareShape(pathEngine, 20));

    std::shared_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/mesh1.tok", buffer);
        mesh = pathEngine.loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
    }

    ZoomExtents(testbed, *mesh); // place camera to see all of newly loaded mesh

    mesh->generateUnobstructedSpaceFor(*agentShape, true, 0);
    mesh->generatePathfindPreprocessFor(*agentShape, 0);

    {
        cAgentAI ai(mesh, agentShape, avoidShape);

        PE::vector<int32_t> selectedAgents;
        PE::vector<bool> agentSelected(ai.numberOfAgents(), false);

        PE::vector<cPosition> waypointsForOrderInProgress;

        cSelectAndMove selectAndMove;

        cMeshRenderGeometry meshRenderGeometry(testbed, *mesh, *agentShape);
        cAgentRenderGeometry agentRenderGeometry(testbed, *agentShape, 20);
        cAgentRenderGeometry highlightMarker(testbed, *avoidShape, 3);

        cMeshLOSPreprocess losPreprocess(*mesh);

        bool paused = false;
        bool doSingleStep = false;
        bool exitFlag = false;
        while(!exitFlag)
        {
        // draw mesh and mesh static elements
            meshRenderGeometry.render(testbed);

        // draw agents
            int32_t i;
            for(i = 0; i != ai.numberOfAgents(); ++i)
            {
                {
                    int32_t groupIndex = ai.agentGroupIndex(i);
                    if(groupIndex == -1)
                    {
                        testbed.setColour("orange");
                    }
                    else
                    {
                        switch(groupIndex % 4)
                        {
                        default:
                            invalid();
                        case 0:
                            testbed.setColour("red");
                            break;
                        case 1:
                            testbed.setColour("purple");
                            break;
                        case 2:
                            testbed.setColour("brown");
                            break;
                        case 3:
                            testbed.setColour("pink");
                            break;
                        }
                    }
                }
                cPosition pos = ai.refAgent(i).getPosition();
                agentRenderGeometry.renderAt(testbed, *mesh, pos);
                if(agentSelected[i])
                {
                    testbed.setColour("lightblue");
                    highlightMarker.renderAt(testbed, *mesh, pos);
                }
            }

            {
                bool completed;
                selectAndMove.update(testbed, completed);
                if(completed)
                {
                    PE::vector<float> screenPositions(ai.numberOfAgents() * 2);
                    for(i = 0; i != ai.numberOfAgents(); ++i)
                    {
                        cPosition p = ai.refAgent(i).getPosition();
                        testbed.worldToScreen(
                            static_cast<float>(p.x),
                            static_cast<float>(p.y),
                            mesh->heightAtPositionF(p),
                            screenPositions[i * 2], screenPositions[i * 2 + 1]);
                    }
                    selectAndMove.updateSelectionSet(testbed, screenPositions, agentSelected, selectedAgents);
                    waypointsForOrderInProgress.clear();
                }
            }

            testbed.setColour("red");
            for(int32_t i = 0; i != SizeL(waypointsForOrderInProgress); ++i)
            {
                cMeshRenderGeometry::Draw3DCross(testbed, *mesh, waypointsForOrderInProgress[i], 20);
            }

            ai.update();

            doSingleStep = false;

            CameraControl(testbed, losPreprocess);

        // tell the testbed to render this frame
            testbed.update(40, exitFlag);

        // receive and process messages for all keys pressed since last frame
            const char* keyPressed;
            while(keyPressed = testbed.receiveKeyMessage())
            {
                {
                    bool handled, moveRequested;
                    selectAndMove.handleInputMessage(testbed, keyPressed, handled, moveRequested);
                    if(handled)
                    {
                        if(moveRequested)
                        {
                            cPosition targetPosition = losPreprocess.positionAtMouse(testbed);
                            if(targetPosition.cell != -1)
                            {
                                targetPosition = mesh->findClosestUnobstructedPosition(*agentShape, 0, targetPosition, clickResolveDistance);
                            }
                            if(testbed.getKeyState("_SHIFT"))
                            {
                                if(targetPosition.cell != -1)
                                {
                                    waypointsForOrderInProgress.push_back(targetPosition);
                                }
                                continue;
                            }
                            assertD(waypointsForOrderInProgress.empty());
                            if(targetPosition.cell == -1)
                            {
                                ai.stopOrder(selectedAgents);
                                continue;
                            }
                            ai.moveOrder(selectedAgents, targetPosition);
                        }
                        continue;
                    }
                }

                if(keyPressed[0] != 'd') // is it a key down message?
                {
                    if(!strcmp("_SHIFT", keyPressed + 1) && !selectAndMove.isActive() && !waypointsForOrderInProgress.empty())
                    {
                        ai.moveOrder(selectedAgents, waypointsForOrderInProgress, waypointsForOrderInProgress.size() > 1);
                        waypointsForOrderInProgress.clear();
                    }
                    continue;
                }

                switch(keyPressed[1])
                {
                case 'P':
                    paused = !paused;
                    break;
                case 'A':
                    doSingleStep = true;
                    break;
                case '_':
                    if(!strcmp("ESCAPE", keyPressed + 2))
                    {
                        exitFlag = true;
                        break;
                    }
                }
            }
        }
    }
    assertR(!mesh->hasRefs());
}
