#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "sampleShared/Error.h"
#include "base/types/Error.h"
#include "sampleShared/MoveAgent.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/CameraControl.h"
#include "common/STL_Helper.h"
#include "base/Container/Vector.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <memory>
#include <string.h>

using PE::vector;
using std::unique_ptr;

void
TestbedApplicationMain(iPathEngine *pathEngine, iTestBed *testbed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testbed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testbed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    bool exitFlag=false;

// create shapes
    unique_ptr<iShape> agent_shape;
    unique_ptr<iShape> swarmer_shape;
    unique_ptr<iShape> danger_shape;
    {
        int32_t array[]=
        {
            8,20,
            20,8,
            20,-8,
            8,-20,
            -8,-20,
            -20,-8,
            -20,8,
            -8,20,
        };
        agent_shape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    {
        int32_t array[]=
        {
            0,16,
            14,-8,
            -14,-8,
        };
        swarmer_shape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    {
        int32_t array[]=
        {
            -60,60,
            60,60,
            60,-60,
            -60,-60,
        };
        danger_shape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    unique_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/mesh1.tok", buffer);
        mesh = pathEngine->loadMeshFromBuffer("tok", VectorBuffer(buffer), SizeU(buffer), 0);
    }

    ZoomExtents(*testbed, *mesh); // place camera to see all of newly loaded mesh

// generate just collision preprocess for the agent shape, and collision and pathfinding preprocess for the swarmers
    mesh->generateUnobstructedSpaceFor(*agent_shape, false, 0);
    mesh->generateUnobstructedSpaceFor(*swarmer_shape, true, 0);
    mesh->generatePathfindPreprocessFor(*swarmer_shape, 0);

    {
    // iAgent objects are used to instantiate shapes on a mesh - either as true agents or as obstructions
    // we initialise values to 0 here to indicate that nothing has yet been placed
        unique_ptr<iAgent> agent;
        vector<unique_ptr<iAgent>> swarmer_agents;

    // paths for swarmers
        vector<unique_ptr<iPath>> swarmer_paths;

    // iCollisionContext objects encapsulate a state of collision on a mesh - essentially this is a collection of agents
    // this context will be used for collision and will contain the agent and all swarmers
        unique_ptr<iCollisionContext> context(mesh->newContext());

    // direction agent is currently facing
        float agent_heading = 0;

    // swarmers take turns to pathfind
    // this variable holds the index of the next swarmer to pathfind
        int32_t next_swarmer = 0;

    // this context will be used for queries around a blocking agent
        unique_ptr<iCollisionContext> temporary_context(mesh->newContext());

        bool swarmers_attack = true;

        cMeshRenderGeometry meshRenderGeometry(*testbed, *mesh);
        cAgentRenderGeometry agentRenderGeometry(*testbed, *agent_shape, 20);
        cAgentRenderGeometry swarmerRenderGeometry(*testbed, *swarmer_shape, 20);

        cMeshLOSPreprocess losPreprocess(*mesh);

        while(!exitFlag)
        {
            // draw mesh and mesh static elements
            meshRenderGeometry.render(*testbed);

        // draw agent if it has been placed
            if(agent)
            {
                testbed->setColour("white");
                agentRenderGeometry.renderAt(*testbed, *mesh, agent->getPosition());
                testbed->setColour("orange");
                cAgentRenderGeometry::DrawAgentHeading(*testbed, *agent, 20, 30, agent_heading);
            }

        // draw any swarmers which have been placed
            testbed->setColour("green");
            for(int32_t i = 0; i < SizeL(swarmer_agents); i++)
            {
                swarmerRenderGeometry.renderAt(*testbed, *mesh, swarmer_agents[i]->getPosition());
            }

            CameraControl(*testbed, losPreprocess);

        // tell the testbed to render this frame
            testbed->update(20, exitFlag);

        // move the agent under the control of user input
            if(agent)
            {
                TurnAgentUnderMouseControl(testbed, agent_heading);
                MoveAgentUnderKeyControl(testbed, pathEngine, agent.get(), agent_heading, context.get(), 6.0f);
            }

        // move swarmers
            if(agent)
            {
                for(int32_t i = 0; i < static_cast<int32_t>(swarmer_agents.size()); i++)
                {
                    if(i == next_swarmer)
                    {
                    // this swarmer is allowed to pathfind this frame

                        if(swarmers_attack)
                        {
                            swarmer_paths[i] = swarmer_agents[i]->findShortestPathTo(0, agent->getPosition());
                        }
                        else
                        {
                            auto danger = mesh->placeAgent(*danger_shape, agent->getPosition());
                            temporary_context->addAgent(*danger);
                            swarmer_paths[i] = swarmer_agents[i]->findPathAway(temporary_context.get(), agent->getPosition(), 400);
                            temporary_context->removeAgent(*danger);
                        }

                        cCollidingLine collidingLine;
                        unique_ptr<iAgent> collidingAgent;
                        float precisionX = 0.f;
                        float precisionY = 0.f;
                        bool collided = swarmer_agents[i]->advanceAlongPathWithCollisionInfo(swarmer_paths[i].get(), 5.f, context.get(), precisionX, precisionY, collidingLine, collidingAgent);
                        if(collided)
                        {
                            if(swarmers_attack && collidingAgent == agent)
                            {
                                collidingAgent = nullptr;
                            }

                            if(collidingAgent)
                            {
                                temporary_context->addAgent(*collidingAgent);

                                uint32_t length_before = swarmer_paths[i]->getLength();

                                if(swarmers_attack)
                                {
                                    swarmer_paths[i] = swarmer_agents[i]->findShortestPathTo(temporary_context.get(), agent->getPosition());
                                }
                                else
                                {
                                    auto danger = mesh->placeAgent(*danger_shape, agent->getPosition());
                                    temporary_context->addAgent(*danger);
                                    swarmer_paths[i] = swarmer_agents[i]->findPathAway(temporary_context.get(), agent->getPosition(), 400);
                                    temporary_context->removeAgent(*danger);
                                }

                                uint32_t length_after = 0;
                                if(swarmer_paths[i])
                                {
                                    length_after = swarmer_paths[i]->getLength();
                                }

                                if(length_after < length_before + 40)
                                {
                                    swarmer_agents[i]->advanceAlongPath(swarmer_paths[i].get(), 5.0f, context.get());
                                }
                                else
                                {
                                    swarmer_paths[i] = nullptr;
                                }

                                temporary_context->removeAgent(*collidingAgent);
                            }
                        }
                    }
                    else
                    {
                    // this swarmer only advances along their current path if possible
                        swarmer_agents[i]->advanceAlongPath(swarmer_paths[i].get(), 5.0f, context.get());
                    }
                }
                next_swarmer++;
                if(next_swarmer >= static_cast<int32_t>(swarmer_agents.size()))
                {
                    next_swarmer = 0;
                }
            }


        // receive and process messages for all keys pressed since last frame
            const char *keypressed;
            while(keypressed=testbed->receiveKeyMessage())
            {
                if(keypressed[0]!='d') // is it a key down message?
                    continue;

                switch(keypressed[1])
                {
                case '_':
                    {
                        if(!strcmp("ESCAPE",keypressed+2))
                        {
                            exitFlag=true;
                            break;
                        }
                        if(!strcmp("SPACE",keypressed+2))
                        {
                            swarmers_attack=!swarmers_attack;
                            break;
                        }
                        break;
                    }
                case '4':
                // place or move the player controlled agent
                    {
                        cPosition p = losPreprocess.positionAtMouse(*testbed);
                        if(p.cell!=-1)
                        {
                        // place the agent if not yet placed, otherwise move to the requested position
                            if(!agent)
                            {
                                if(!mesh->testPointCollision(*agent_shape, context.get(), p))
                                {
                                    agent = mesh->placeAgent(*agent_shape, p);
                                    context->addAgent(*agent);
                                }
                            }
                            else
                            {
                                if(!agent->testCollisionAt(context.get(), p))
                                    agent->moveTo(p);
                            }
                        }
                    }
                    break;
                case '5':
                // place a new swarmer
                    if(swarmer_agents.size() < 30)
                    {
                    // get the position on the mesh under the mouse
                    // p.cell is set to -1 if the mouse is not over the mesh
                        cPosition p = losPreprocess.positionAtMouse(*testbed);
                        if(p.cell!=-1 && !mesh->testPointCollision(*swarmer_shape, context.get(), p))
                        {
                            swarmer_agents.emplace_back(mesh->placeAgent(*swarmer_shape,p));
                            context->addAgent(*swarmer_agents.back());
                            swarmer_paths.emplace_back(nullptr);
                        }
                    }
                    break;
                }
            }
        }
    }
    assertR(!mesh->hasRefs());
}
