#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "sampleShared/Error.h"
#include "base/types/Error.h"
#include "sampleShared/MoveAgent.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "sampleShared/CameraControl.h"
#include "common/STL_Helper.h"
#include "base/Container/Vector.h"
#include "platform_common/StrCmpI.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <sstream>
#include <memory>
#include <math.h>

using PE::vector;
using std::unique_ptr;

void
TestbedApplicationMain(iPathEngine *pathEngine, iTestBed *testbed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testbed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testbed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    bool exitFlag=false;
    
// create shapes
    unique_ptr<iShape> agent_shape;
    vector<unique_ptr<iShape>> obstruction_shapes;
    {
        int32_t array[]=
        {
            8,20,
            20,8,
            20,-8,
            8,-20,
            -8,-20,
            -20,-8,
            -20,8,
            -8,20,
        };
        agent_shape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    {
        int32_t array[]=
        {
            30,30,
            30,-30,
            -30,-30,
            -30,30,
        };
        obstruction_shapes.push_back(pathEngine->newShape(array, sizeof(array) / sizeof(*array)));
    }
    {
        int32_t array[]=
        {
            10,35,
            20,-20,
            -10,-30,
            -15,32,
        };
        obstruction_shapes.push_back(pathEngine->newShape(array, sizeof(array) / sizeof(*array)));
    }
    {
        int32_t array[]=
        {
            15,60,
            50,0,
            10,-40,
            -40,-40,
            -40,-5,
        };
        obstruction_shapes.push_back(pathEngine->newShape(array, sizeof(array) / sizeof(*array)));
    }
    {
        int32_t array[]=
        {
            15,20,
            18,-5,
            7,-20,
            -18,-18,
            -16,16,
        };
        obstruction_shapes.push_back(pathEngine->newShape(array, sizeof(array) / sizeof(*array)));
    }
    obstruction_shapes.push_back(nullptr);

    unique_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/mesh1.xml", buffer);
        mesh = pathEngine->loadMeshFromBuffer("xml", VectorBuffer(buffer), SizeU(buffer), 0);
    }

    ZoomExtents(*testbed, *mesh); // place camera to see all of newly loaded mesh

// generate preprocess for the agent shape
    mesh->generateUnobstructedSpaceFor(*agent_shape, true, 0);
    mesh->generatePathfindPreprocessFor(*agent_shape, 0);

// iAgent objects are used to instantiate shapes on a mesh - either as true agents or as obstructions

    // agent pointer are left initially unset, to indicate that nothing has yet been placed
    unique_ptr<iAgent> agent;
    float agentPrecisionX = 0.f, agentPrecisionY = 0.f;
    vector<unique_ptr<iAgent>> obstruction_agents(obstruction_shapes.size() - 1);

// iCollisionContext objects encapsulate a state of collision on a mesh - essentially this is a collection of agents
    unique_ptr<iCollisionContext> context(mesh->newContext());

// target position for pathfinding
    cPosition agent_target;
// path to target
    unique_ptr<iPath> path;

// direction agent is currently facing
    float agent_heading=0;

    bool drawExpansion = false;
    bool paused = false;

    cMeshRenderGeometry meshRenderGeometry(*testbed, *mesh);
    auto unobstructedSpaceBoundaries = testbed->newRenderGeometry();
    cMeshRenderGeometry::AddUnobstructedSpaceBoundaries(*mesh, *agent_shape, *unobstructedSpaceBoundaries);
    cAgentRenderGeometry agentRenderGeometry(*testbed, *agent_shape, 70);

    cMeshLOSPreprocess losPreprocess(*mesh);

    while(!exitFlag)
    {
        // draw mesh and mesh static elements
        meshRenderGeometry.render(*testbed);

        if(drawExpansion)
        {
            testbed->setColour("orange");
            testbed->render(*unobstructedSpaceBoundaries);
        }

    // determine closest unobstructed point to agent target if necessary,
    // draw target and closest unobstructed point as appropriate
        cPosition agent_target_closest_unobstructed_point;
        if(agent_target.cell!=-1)
        {
            if(mesh->testPointCollision(*agent_shape, context.get(), agent_target))
            {
                testbed->setColour("red");
                cMeshRenderGeometry::Draw3DCross(*testbed, *mesh, agent_target, 20);
                agent_target_closest_unobstructed_point=mesh->findClosestUnobstructedPosition(
                                    *agent_shape,
                                    context.get(),
                                    agent_target,
                                    20);
                testbed->setColour("orange");
                if(agent_target_closest_unobstructed_point.cell != -1)
                    cMeshRenderGeometry::Draw3DCross(*testbed, *mesh, agent_target_closest_unobstructed_point, 15);
            }
            else
            {
                testbed->setColour("orange");
                cMeshRenderGeometry::Draw3DCross(*testbed, *mesh, agent_target, 30);
                agent_target_closest_unobstructed_point = agent_target;
            }
        }

    // draw agent if it has been placed
        if(agent)
        {
            testbed->setColour("white");
            agentRenderGeometry.renderAt(*testbed, *mesh, agent->getPosition(), agentPrecisionX, agentPrecisionY);
            testbed->setColour("orange");
            cAgentRenderGeometry::DrawAgentHeading(*testbed, *agent, 70.f, 30.f, agent_heading, agentPrecisionX, agentPrecisionY);

        // pathfind to target if it has been placed
            if(agent_target_closest_unobstructed_point.cell!=-1 && path==0)
            {
                path = agent->findShortestPathTo(context.get(), agent_target_closest_unobstructed_point);
            }
        }

    // draw path
        testbed->setColour("green");
        if(path)
            cMeshRenderGeometry::DrawPath(*testbed, *path);

    // draw any obstacles which have been placed
        for(uint32_t i = 0; i < obstruction_agents.size(); i++)
        {
            if(obstruction_agents[i])
            {
                float height = 20;
                if(obstruction_agents[i]->getTraverseCost() == -1.f)
                {
                    testbed->setColour("red");
                }
                else
                {
                    testbed->setColour("purple");
                    height = 4;
                }
                unique_ptr<iRenderGeometry> rg = testbed->newRenderGeometry();
                cAgentRenderGeometry::AddCylinder(*obstruction_agents[i], height, *rg);
                testbed->render(*rg);
                if(drawExpansion)
                {
                    testbed->setColour("orange");
                    unique_ptr<iRenderGeometry> rg = testbed->newRenderGeometry();
                    cAgentRenderGeometry::AddAgentExpansion(*obstruction_agents[i], *agent_shape, *rg);
                    testbed->render(*rg);
                }
            }
        }

        CameraControl(*testbed, losPreprocess);

    // tell the testbed to render this frame
        testbed->update(40, exitFlag);

        if(agent)
        {
            if(path && !paused)
            {
                if(path->size()>=2)
                {
                // set heading for the current path section
                    cPosition next_target=path->position(1);
                    cPosition current=agent->getPosition();
                    int32_t dx,dy;
                    dx=next_target.x-current.x;
                    dy=next_target.y-current.y;
                    agent_heading = static_cast<float>(atan2(static_cast<double>(dx),static_cast<double>(dy)));
                }
                agent->advanceAlongPathWithPrecision(path.get(), 11.f, context.get(), agentPrecisionX, agentPrecisionY);
            }
            else
            {
                TurnAgentUnderMouseControl(testbed,agent_heading);
                MoveAgentUnderKeyControl(testbed, pathEngine,
                                        agent.get(), agent_heading, context.get(), 7.f,
                                        agentPrecisionX, agentPrecisionY);
            }
        }

    // receive and process messages for all keys pressed since last frame
        const char *keypressed;
        while(keypressed=testbed->receiveKeyMessage())
        {
            if(keypressed[0]!='d') // is it a key down message?
                continue;

            if(keypressed[1]>='6' && keypressed[1]<='9')
            {
            // move obstruction shape
                int32_t index = keypressed[1]-'6';
                if(static_cast<size_t>(index) < obstruction_agents.size())
                {
                // get the position on the mesh under the mouse
                // p.cell t set to -1 if the mouse is not over the mesh
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1 &&
                        (agent==0 || !agent->testCollisionDirectlyAgainstPlacedShape(*obstruction_shapes[index], p))
                        )
                    {
                    // place the obstruction if not yet placed, otherwise move to the requested position
                        if(!obstruction_agents[index])
                        {
                            obstruction_agents[index] = mesh->placeAgent(*obstruction_shapes[index], p);
                            context->addAgent(*obstruction_agents[index]);
                        }
                        else
                        {
                            obstruction_agents[index]->moveTo(p);
                        }                        
                    // holding the left shift key makes the obstacle a 'soft obstacle'
                        if(testbed->getKeyState("_SHIFT"))
                        {
                            // note that the cost is in addition to base movement cost
                            // so movement through the obstacle will cost twice as much as movement around
                            obstruction_agents[index]->setTraverseCost(1.f);
                        }
                        else
                        {
                            obstruction_agents[index]->setTraverseCost(-1.f);
                        }
                    // the current path may need to be regenerated
                        path = nullptr;
                    }
                }
                continue;
            }

            switch(keypressed[1])
            {
            case '_':
                {
                    if(!StrCmpI("ESCAPE", keypressed + 2))
                        exitFlag=true;
                    break;
                }
            case '4':
                {
                // move or place the agent
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1 && !mesh->testPointCollision(*agent_shape, context.get(), p))
                    {
                        if(!agent)
                            agent = mesh->placeAgent(*agent_shape, p);
                        else
                            agent->moveTo(p);
                        agent_target.cell = -1;
                        path = nullptr;
                    }
                }
                break;
            case '5':
                {
                // set a target for pathfinding
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1)
                        agent_target=p;
                    path = nullptr;
                }
                break;
            case 'W':
            case 'A':
            case 'S':
            case 'D':
            // a direct control key has been pressed
            // delete path
            // control will revert to user control
                agent_target.cell = -1;
                path = nullptr;
                break;
            case 'E':
                drawExpansion = !drawExpansion;
                break;
            case 'P':
                paused = !paused;
                break;
            }
        }
    }
}
