#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "project/testbedApp/ThreadPerMesh/MeshAndAgents.h"
#include "sampleShared/CameraControl.h"
#include "sampleShared/Error.h"
#include "base/Container/Vector.h"
#include "base/types/Error.h"
#include "sampleShared/LoadBinary.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "platform_common/Thread.h"
#include "externalAPI/i_pathengine.h"
#include <memory>
#include <string>
#include <string.h>
#include <sstream>
#include <algorithm>

using PE::vector;
using std::unique_ptr;

namespace
{
class cThread : public iThread
{
    cMeshAndAgents* _meshAndAgents;
public:
    cThread(cMeshAndAgents* meshAndAgents)
    {
        _meshAndAgents = meshAndAgents;
    }
    ~cThread()
    {
        delete _meshAndAgents;
    }
    void
    run()
    {
        _meshAndAgents->run();
    }
};
} // anonymous namespace

static void
AddInstance(
        iPathEngine& pathEngine, iTestBed& testBed, iShape& agentShape,
        const vector<char>& meshBuffer,
        vector<cMeshAndAgents*>& instances,
        vector<unique_ptr<iThread>>& threads,
        vector<tThreadHandle>& threadHandles        
        )
{
    cMeshAndAgents* instance = new cMeshAndAgents(pathEngine, testBed, agentShape, meshBuffer);
    cThread* thread = new cThread(instance);
    instances.push_back(instance);
    threads.emplace_back(thread);
    threadHandles.push_back(StartThread(*thread));
}

void
TestbedApplicationMain(iPathEngine* pathEngine, iTestBed* testBed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testBed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testBed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testBed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    unique_ptr<iShape> agentShape;
    {
        int32_t array[]=
        {
            -30, 30,
            30, 30,
            30, -30,
            -30, -30,
        };
        agentShape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

    vector<vector<char>> meshBuffers;

    meshBuffers.emplace_back();
    LoadBinary("../resource/meshes/demo_level1.tok", meshBuffers.back());

    meshBuffers.emplace_back();
    LoadBinary("../resource/meshes/demo_level2.tok", meshBuffers.back());

    meshBuffers.emplace_back();
    LoadBinary("../resource/meshes/demo_biglevel.tok", meshBuffers.back());

    meshBuffers.emplace_back();
    LoadBinary("../resource/meshes/demo_triviallevel.tok", meshBuffers.back());


    vector<cMeshAndAgents*> instances;
    vector<unique_ptr<iThread>> threads;
    vector<tThreadHandle> threadHandles;
    
    AddInstance(*pathEngine, *testBed, *agentShape, meshBuffers[0], instances, threads, threadHandles);
    AddInstance(*pathEngine, *testBed, *agentShape, meshBuffers[1], instances, threads, threadHandles);
    AddInstance(*pathEngine, *testBed, *agentShape, meshBuffers[2], instances, threads, threadHandles);
    AddInstance(*pathEngine, *testBed, *agentShape, meshBuffers[3], instances, threads, threadHandles);
    AddInstance(*pathEngine, *testBed, *agentShape, meshBuffers[2], instances, threads, threadHandles);

    int32_t currentInstance = 0;

    bool changedInstance = true;
    bool exitFlag = false;
    while(!exitFlag)
    {
        CameraControl(*testBed);
        testBed->update(20, exitFlag);

        instances[currentInstance]->render(*testBed, currentInstance, changedInstance);
        changedInstance = false;

        testBed->setColour("white");
        testBed->printTextLine(0, "(change instance with up and down arrow keys)");
        testBed->printTextLine(0, "(press 'a' to add an agent to the currently selected instance)");

        for(int32_t i = 0; i != SizeL(instances); ++i)
        {
            std::ostringstream oss;
            oss << "instance " << i << ": " << instances[i]->getQueryCount() << " queries";
            testBed->setColour(i == currentInstance ? "green" : "orange");
            testBed->printTextLine(0, oss.str().c_str());
        }

    // receive and process messages for all keys pressed since last frame
        const char *keypressed;
        while(keypressed = testBed->receiveKeyMessage())
        {
            if(keypressed[0] != 'd') // is it a key down message?
                continue;

            switch(keypressed[1])
            {
            case '_':
                {
                    if(!strcmp("ESCAPE", keypressed + 2))
                    {
                        exitFlag = true;
                        break;
                    }
                    if(!strcmp("UP", keypressed + 2))
                    {
                        if(currentInstance + 1 < static_cast<int32_t>(instances.size()))
                        {
                            ++currentInstance;
                            changedInstance = true;
                        }
                        break;
                    }
                    if(!strcmp("DOWN", keypressed + 2))
                    {
                        if(currentInstance)
                        {
                            --currentInstance;
                            changedInstance = true;
                        }
                        break;
                    }
                    break;
                }
            case 'A':
                instances[currentInstance]->addAgent();
                break;
            }
        }
    }

    for(int32_t i = 0; i != SizeL(instances); ++i)
    {
        instances[i]->terminate();
    }
    for(int32_t i = 0; i != SizeL(threads); ++i)
    {
        JoinThread(threadHandles[i]);
    }
}
