#include "base/types/Header.h"
#include "project/testbedApp/ThreadPerMesh/MeshAndAgents.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "common/STL_Helper.h"
#include "base/MutexLock.h"
#include "platform_common/Thread.h"
#include "externalAPI/i_pathengine.h"
#include <vector>

void
cMeshAndAgents::generatePreprocessIfNotPresent()
{
    if(!_mesh->shapeCanPathfind(*_agentShape, 0))
    {
        const char* options[3];
        options[0] = "splitWithCircumferenceBelow";
        options[1] = "2000";
        options[2] = 0;
        //.... load preprocess from persistence, normally, for best performance
        _mesh->generateUnobstructedSpaceFor(*_agentShape, true, options);
        _mesh->generateConnectedRegionPreprocessFor(*_agentShape, 0);
        _mesh->generatePathfindPreprocessFor(*_agentShape, 0);
    }
}

void
cMeshAndAgents::doAddAgent()
{
    generatePreprocessIfNotPresent();
    cPosition p;
    do
    {
        p = _mesh->generateRandomPosition();
    }
    while(_mesh->testPointCollision(*_agentShape, 0, p));
    _agents.emplace_back(_mesh->placeAgent(*_agentShape, p));
    _paths.emplace_back(nullptr);
}

void
cMeshAndAgents::updateAgent(int32_t i)
{
    if(_paths[i])
    {
        bool collided = _agents[i]->advanceAlongPath(_paths[i].get(), 12.f, 0);
        assertD(!collided);
        if(_paths[i]->size() < 2)
        {
            _paths[i] = nullptr;
        }
        return;
    }
// attempt to generate a new path
    cPosition p = _mesh->generateRandomPosition();
    if(_mesh->getConnectedRegionFor(*_agentShape, p) != _mesh->getConnectedRegionFor(*_agentShape, _agents[i]->getPosition()))
    {
        return;
    }
    {
        cMutexLock lock(_queryCountMutex);
        ++_queryCount;
    }
    _paths[i] = _agents[i]->findShortestPathTo(0, p);
}

cMeshAndAgents::cMeshAndAgents(
        iPathEngine& pathEngine,
        iTestBed& testBed,
        iShape& agentShape,
        const PE::vector<char>& meshBuffer
        ) :
    _mesh(pathEngine.loadMeshFromBuffer("tok", VectorBuffer(meshBuffer), SizeU(meshBuffer), 0)),
    _meshRenderGeometry(testBed, *_mesh),
    _agentRenderGeometry(testBed, agentShape, 120)

{
    _agentShape = agentShape.addExternalRef();
    _queryCount = 0;
    _agentsToAdd = 0;
    _terminate = false;
}

void
cMeshAndAgents::run()
{
    bool terminate;
    do
    {
        {
            cMutexLock lock(_positionsMutex);
            _positionsBeforeUpdate.resize(_agents.size());
            for(int32_t i = 0; i != _agents.size(); ++i)
            {
                _positionsBeforeUpdate[i] = _agents[i]->getPosition();
            }
        }

        {
            int32_t agentsToAdd;
            {
                cMutexLock lock(_agentAddMutex);
                agentsToAdd = _agentsToAdd;
                _agentsToAdd = 0;
            }
            for(int32_t i = 0; i != agentsToAdd; ++i)
            {
                doAddAgent();
            }
        }

        for(int32_t i = 0; i != _agents.size(); ++i)
        {
            updateAgent(i);
        }

        Sleep_Milliseconds(1);

        cMutexLock lock(_terminateMutex);
        terminate = _terminate;
    }
    while(!terminate);
}

void
cMeshAndAgents::addAgent()
{
    cMutexLock lock(_agentAddMutex);
    ++_agentsToAdd;
}

void
cMeshAndAgents::render(iTestBed& testBed, int32_t meshSlot, bool initCamera) const
{
    if(initCamera)
    {
        ZoomExtents(testBed, *_mesh);
    }
    _meshRenderGeometry.render(testBed);
    PE::vector<cPosition> positions;
    {
        cMutexLock lock(_positionsMutex);
        positions = _positionsBeforeUpdate;
    }
    testBed.setColour("orange");
    for(int32_t i = 0; i != positions.size(); ++i)
    {
        _agentRenderGeometry.renderAt(testBed, *_mesh, positions[i]);
    }
}

int32_t
cMeshAndAgents::getQueryCount() const
{
    cMutexLock lock(_queryCountMutex);
    return _queryCount;
}

void
cMeshAndAgents::terminate()
{
    cMutexLock lock(_terminateMutex);
    _terminate = true;
}
