#include "projectCommon/SingleObject/DirectGlobalAllocator.h"
#include "base/types/Header.h"
#include "sampleShared/CameraControl.h"
#include "sampleShared/Error.h"
#include "base/types/Error.h"
#include "sampleShared/LoadBinary.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "common/STL_Helper.h"
#include "base/Container/Vector.h"
#include "platform_common/TestbedApplicationEntryPoint.h"
#include "externalAPI/i_pathengine.h"
#include <string.h>
#include <memory>

using std::unique_ptr;

void
TestbedApplicationMain(iPathEngine *pathEngine, iTestBed *testbed)
{
// check if interfaces are compatible with the headers used for compilation
    if(testbed->getInterfaceMajorVersion()!=TESTBED_INTERFACE_MAJOR_VERSION
        ||
        testbed->getInterfaceMinorVersion()<TESTBED_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Testbed version is incompatible with headers used for compilation.");
        return;
    }
    if(pathEngine->getInterfaceMajorVersion()!=PATHENGINE_INTERFACE_MAJOR_VERSION
        ||
        pathEngine->getInterfaceMinorVersion()<PATHENGINE_INTERFACE_MINOR_VERSION)
    {
        testbed->reportError("Fatal","Pathengine version is incompatible with headers used for compilation.");
        return;
    }

    unique_ptr<iMesh> mesh;
    {
        PE::vector<char> buffer;
        LoadBinary("../resource/meshes/mesh1.xml", buffer);
        mesh = pathEngine->loadMeshFromBuffer("xml", VectorBuffer(buffer), SizeU(buffer), 0);
    }

    ZoomExtents(*testbed, *mesh);

// create shapes
    unique_ptr<iShape> agent_shape;
    unique_ptr<iShape> obstruction_shape1;
    unique_ptr<iShape> obstruction_shape2;
    {
        int32_t array[]=
        {
            8,20,
            20,8,
            20,-8,
            8,-20,
            -8,-20,
            -20,-8,
            -20,8,
            -8,20,
        };
        agent_shape = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    {
        int32_t array[]=
        {
            30,30,
            30,-30,
            -30,-30,
            -30,30,
        };
        obstruction_shape1 = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }
    {
        int32_t array[]=
        {
            10,35,
            20,-20,
            -10,-30,
            -15,32,
        };
        obstruction_shape2 = pathEngine->newShape(array, sizeof(array) / sizeof(*array));
    }

// generate collision preprocess for the agent shape
    mesh->generateUnobstructedSpaceFor(*agent_shape, false, 0);

// iAgent objects are used to instantiate shapes on a mesh - either as true agents or as obstructions
// agent pointers are left unset here to indicate that nothing has yet been placed
    unique_ptr<iAgent> agent;
    unique_ptr<iAgent> obstruction_agent1;
    unique_ptr<iAgent> obstruction_agent2;

// iCollisionContext objects encapsulate a state of collision on a mesh - essentially this is a set of agents
    unique_ptr<iCollisionContext> context(mesh->newContext());

    cMeshRenderGeometry meshRenderGeometry(*testbed, *mesh);
    cAgentRenderGeometry agentRenderGeometry(*testbed, *agent_shape, 20);
    cAgentRenderGeometry obstruction1RenderGeometry(*testbed, *obstruction_shape1, 20);
    cAgentRenderGeometry obstruction2RenderGeometry(*testbed, *obstruction_shape2, 20);

    cMeshLOSPreprocess losPreprocess(*mesh);

    bool exitFlag = false;
    while(!exitFlag)
    {
    // draw mesh and mesh static elements
        meshRenderGeometry.render(*testbed);

    // draw agent if it has been placed
        if(agent)
        {
            testbed->setColour("white");
            agentRenderGeometry.renderAt(*testbed, *mesh, agent->getPosition());
        }

    // draw any obstructions that have been placed
        if(obstruction_agent1)
        {
            testbed->setColour("red");
            obstruction1RenderGeometry.renderAt(*testbed, *mesh, obstruction_agent1->getPosition());
        }
        if(obstruction_agent2)
        {
            testbed->setColour("red");
            obstruction2RenderGeometry.renderAt(*testbed, *mesh, obstruction_agent2->getPosition());
        }

        CameraControl(*testbed, losPreprocess);
        testbed->update(20, exitFlag);

    // receive and process messages for all keys pressed since last frame
        const char *keypressed;
        while(keypressed=testbed->receiveKeyMessage())
        {
            if(keypressed[0]!='d') // is it a key down message?
                continue;

            switch(keypressed[1])
            {
            case '_':
                {
                    if(strcmp("ESCAPE",keypressed+2)==0)
                        exitFlag=true;
                    break;
                }
            case '4':
            // place or move the agent
                {
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1 && !mesh->testPointCollision(*agent_shape, context.get(), p))
                    {
                        if(!agent)
                            agent = mesh->placeAgent(*agent_shape, p);
                        else
                            agent->moveTo(p);
                    }
                }
                break;
            case '5':
            // place or move obstruction 1
                {
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1 && (agent==0 || !agent->testCollisionDirectlyAgainstPlacedShape(*obstruction_shape1, p)))
                    {
                    // place the agent if not yet placed, otherwise move to the requested position
                        if(!obstruction_agent1)
                        {
                            obstruction_agent1 = mesh->placeAgent(*obstruction_shape1, p);
                            context->addAgent(*obstruction_agent1);
                        }
                        else
                            obstruction_agent1->moveTo(p);
                    }
                }
                break;
            case '6':
            // place or move obstruction 2
                {
                    cPosition p = losPreprocess.positionAtMouse(*testbed);
                    if(p.cell!=-1 && (agent==0 || !agent->testCollisionDirectlyAgainstPlacedShape(*obstruction_shape2, p)))
                    {
                    // place the agent if not yet placed, otherwise move to the requested position
                        if(!obstruction_agent2)
                        {
                            obstruction_agent2 = mesh->placeAgent(*obstruction_shape2, p);
                            context->addAgent(*obstruction_agent2);
                        }
                        else
                            obstruction_agent2->moveTo(p);
                    }
                }
                break;
            }
        }
    }
}
