#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/RenderGeometryRenderCallBack.h"
#include "base/Container/Vector.h"
#include "base/types/Assert.h"
#include "externalAPI/i_pathengine.h"
#include <math.h>

using PE::vector;
using std::unique_ptr;

void
cAgentRenderGeometry::AddCylinder(const vector<float>& shapeCoords, float startZ, float endZ, iRenderGeometry& addTo)
{
    uint32_t coordsSize = static_cast<uint32_t>(shapeCoords.size());
    assertD((coordsSize % 2) == 0);
    {
        float data[4 * 3];
        data[0 * 3 + 2] = startZ;
        data[3 * 3 + 2] = startZ;
        data[2 * 3 + 2] = endZ;
        data[1 * 3 + 2] = endZ;
        for(uint32_t i = 0; i < coordsSize; i += 2)
        {
            float x = shapeCoords[i];
            float y = shapeCoords[i + 1];
            uint32_t nextI = ((i + 2) % coordsSize);
            float nextX = shapeCoords[nextI];
            float nextY = shapeCoords[nextI + 1];
            data[0 * 3 + 0] = nextX;
            data[0 * 3 + 1] = nextY;
            data[1 * 3 + 0] = nextX;
            data[1 * 3 + 1] = nextY;
            data[3 * 3 + 0] = x;
            data[3 * 3 + 1] = y;
            data[2 * 3 + 0] = x;
            data[2 * 3 + 1] = y;
            addTo.addPolygon(data, 4 * 3);
        }
    }
    vector<float> data;
    for(uint32_t i = 0; i < coordsSize; i += 2)
    {
        data.push_back(shapeCoords[i]);
        data.push_back(shapeCoords[i + 1]);
        data.push_back(endZ);
    }
    addTo.addPolygon(&data.front(), static_cast<uint32_t>(data.size()));
}
void
cAgentRenderGeometry::AddCylinder(const iShape& shape, float height, iRenderGeometry& addTo)
{
    vector<float> shapeCoords;
    for(int32_t i = 0; i < shape.size(); i++)
    {
        int32_t x, y;
        shape.vertex(i, x, y);
        shapeCoords.push_back(static_cast<float>(x));
        shapeCoords.push_back(static_cast<float>(y));
    }
    AddCylinder(shapeCoords, 0.f, height, addTo);
}
void
cAgentRenderGeometry::AddCylinder(const iAgent& agent, float height, iRenderGeometry& addTo)
{
    vector<float> shapeCoords;
    for(int32_t i = 0; i < agent.getNumberOfVertices(); i++)
    {
        int32_t x, y;
        agent.getVertexCoordinates(i, x, y);
        shapeCoords.push_back(static_cast<float>(x));
        shapeCoords.push_back(static_cast<float>(y));
    }
    float groundHeight = agent.refMesh().heightAtPositionF(agent.getPosition());
    AddCylinder(shapeCoords, groundHeight, groundHeight + height, addTo);
}
void
cAgentRenderGeometry::AddAgentExpansion(const iAgent& agent, const iShape& expandShape, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    agent.refMesh().renderAgentExpansion(expandShape, agent, 0, 0, callBack);
}

void
cAgentRenderGeometry::AddDirectionArrowRenderGeometry(const iAgent& agent, float height, float size, float direction, float precisionX, float precisionY, iRenderGeometry& addTo)
{
    cPosition p = agent.getPosition();
    float z = agent.refMesh().heightAtPositionF(p) + height;

    float data[3 * 3];
    float sinof, cosof;
    sinof = static_cast<float>(sin(double(direction)));
    cosof = static_cast<float>(cos(double(direction)));

    data[0 * 3 + 0] = p.x + size * sinof + precisionX;
    data[0 * 3 + 1] = p.y + size * cosof + precisionY;
    data[0 * 3 + 2] = z;

    data[1 * 3 + 0] = p.x + size * cosof*0.25f + precisionX;
    data[1 * 3 + 1] = p.y - size * sinof*0.25f + precisionY;
    data[1 * 3 + 2] = z;

    data[2 * 3 + 0] = p.x - size * cosof*0.25f + precisionX;
    data[2 * 3 + 1] = p.y + size * sinof*0.25f + precisionY;
    data[2 * 3 + 2] = z;

    addTo.addPolygon(data, 3 * 3);
}

void
cAgentRenderGeometry::DrawAgentHeading(iTestBed& testBed, const iAgent& agent, float height, float size, float direction, float precisionX, float precisionY)
{
    auto rg = testBed.newRenderGeometry();
    AddDirectionArrowRenderGeometry(agent, height, size, direction, precisionX, precisionY, *rg);
    testBed.render(*rg);
}

cAgentRenderGeometry::cAgentRenderGeometry(iTestBed& testBed, const iShape& shape, float height)
{
    _geometry = testBed.newRenderGeometry();
    AddCylinder(shape, height, *_geometry);
}

void
cAgentRenderGeometry::renderAt(iTestBed& testBed, iMesh& mesh, cPosition position) const
{
    testBed.render_Offset(
        *_geometry,
        static_cast<float>(position.x),
        static_cast<float>(position.y),
        mesh.heightAtPositionF(position)
        );
}
void
cAgentRenderGeometry::renderAt(iTestBed& testBed, iMesh& mesh, cPosition position, float precisionX, float precisionY) const
{
    testBed.render_Offset(
        *_geometry,
        static_cast<float>(position.x) + precisionX,
        static_cast<float>(position.y) + precisionY,
        mesh.heightAtPositionF(position)
        );
}
