//**********************************************************************
//
// Copyright (c) 2002-2012
// PathEngine
// Lyon, France
//
// All Rights Reserved
//
//**********************************************************************

#include "base/types/Header.h"
#include "sampleShared/MeshLOSPreprocess.h"
#include "base/Geometry/Vector3F.h"
#include "externalAPI/i_pathengine.h"

using PE::vector;

static cVector3F
V3FFor(const iMesh& mesh, int32_t f, int32_t e)
{
    int32_t x, y;
    float z;
    mesh.get3DFaceVertex(f, e, x, y, z);
    return cVector3F(static_cast<float>(x), static_cast<float>(y), z);
}

cMeshLOSPreprocess::cMeshLOSPreprocess(const iMesh& mesh) : _mesh(mesh.addConstExternalRef())
{
    for(int32_t f = 0; f != mesh.getNumberOf3DFaces(); ++f)
    {
        cVector3F v1 = V3FFor(mesh, f, 0);
        cVector3F v2 = V3FFor(mesh, f, 1);
        cVector3F v3 = V3FFor(mesh, f, 2);
        _tris.push_back(nLineCollidesWithTriangle::cTriPreprocess(v1, v2, v3));
    }
}

bool
cMeshLOSPreprocess::lineCollides(const nLineCollidesWithTriangle::cLinePreprocess& line, int32_t& faceIndex) const
{
    for(int32_t i = 0; i < SizeL(_tris); i++)
    {
        if(nLineCollidesWithTriangle::lineCollidesWithTriangle(_tris[i], line))
        {
            faceIndex = static_cast<int32_t>(i);
            return true;
        }
    }
    return false;
}

bool
cMeshLOSPreprocess::rayCollides(const nLineCollidesWithTriangle::cLinePreprocess& line, int32_t& faceIndex, float& fractionAlong) const
{
    faceIndex = -1;
    for(int32_t i = 0; i < SizeL(_tris); i++)
    {
        float candidateFraction;
        if(nLineCollidesWithTriangle::rayCollidesWithTriangle(_tris[i], line, candidateFraction))
        {
            if(faceIndex == -1 || candidateFraction < fractionAlong)
            {
                faceIndex = static_cast<int32_t>(i);
                fractionAlong = candidateFraction;
            }
        }
    }
    return faceIndex != -1;
}
void
cMeshLOSPreprocess::getFacesObstructingLine(const nLineCollidesWithTriangle::cLinePreprocess& line,
    vector<int32_t>& appendTo) const
{
    for(int32_t i = 0; i < SizeL(_tris); i++)
    {
        if(nLineCollidesWithTriangle::lineCollidesWithTriangle(_tris[i], line))
        {
            appendTo.push_back(static_cast<int32_t>(i));
        }
    }
}

bool
cMeshLOSPreprocess::pickPosition(
    const cVector3F& cameraPosition, const cVector3F& targetScreenPosition,
    cPosition& result,
    cVector3F& approximatePoint
) const
{
    int32_t face;
    nLineCollidesWithTriangle::cLinePreprocess line(cameraPosition, targetScreenPosition);
    float fractionAlong;
    if(!rayCollides(line, face, fractionAlong))
    {
        return false;
    }

    approximatePoint = (targetScreenPosition - cameraPosition) * fractionAlong + cameraPosition;
    int32_t x = static_cast<int32_t>(approximatePoint.x());
    int32_t y = static_cast<int32_t>(approximatePoint.y());

    if(_mesh->pointIsIn3DFace(face, x, y))
    {
        result = _mesh->positionForPointIn3DFace(face, x, y);
        return true;
    }
    if(_mesh->pointIsIn3DFace(face, x + 1, y))
    {
        result = _mesh->positionForPointIn3DFace(face, x + 1, y);
        return true;
    }
    if(_mesh->pointIsIn3DFace(face, x - 1, y))
    {
        result = _mesh->positionForPointIn3DFace(face, x - 1, y);
        return true;
    }
    if(_mesh->pointIsIn3DFace(face, x, y + 1))
    {
        result = _mesh->positionForPointIn3DFace(face, x, y + 1);
        return true;
    }
    if(_mesh->pointIsIn3DFace(face, x, y - 1))
    {
        result = _mesh->positionForPointIn3DFace(face, x, y - 1);
        return true;
    }
    return false;
}
bool
cMeshLOSPreprocess::positionAtMouse(
    iTestBed& testBed,
    cPosition& result,
    cVector3F& approximatePoint
) const
{
    cVector3F cameraPosition;
    testBed.getCameraPosition(cameraPosition.x(), cameraPosition.y(), cameraPosition.z());
    int32_t mouseX_I, mouseY_I;
    testBed.getMouseScreenPosition(mouseX_I, mouseY_I);
    float mouseX = static_cast<float>(mouseX_I);
    float mouseY = static_cast<float>(mouseY_I);
    cVector3F targetScreenPosition;
    testBed.screenToWorld(mouseX, mouseY, targetScreenPosition.x(), targetScreenPosition.y(), targetScreenPosition.z());
    return pickPosition(cameraPosition, targetScreenPosition, result, approximatePoint);
}

cPosition
cMeshLOSPreprocess::positionAtMouse(iTestBed& testBed) const
{
    cPosition result;
    cVector3F approximatePoint_Ignored;
    bool success = positionAtMouse(testBed, result, approximatePoint_Ignored);
    if(!success)
        result.cell = -1;
    return result;
}
bool
cMeshLOSPreprocess::positionAtMouse(iTestBed& testBed, cVector3F& result) const
{
    cPosition meshPosition_Ignored;
    return positionAtMouse(testBed, meshPosition_Ignored, result);
}
