#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/RenderGeometryRenderCallBack.h"
#include "externalAPI/i_pathengine.h"
#include <vector>

using PE::vector;
using std::unique_ptr;

static void
GetVertPoint(const iMesh& mesh, int32_t face, int32_t vertInFace, float*& ptr)
{
    int32_t x, y;
    float z;
    mesh.get3DFaceVertex(face, vertInFace, x, y, z);
    *ptr++ = static_cast<float>(x);
    *ptr++ = static_cast<float>(y);
    *ptr++ = z;
}

void
cMeshRenderGeometry::AddMeshTris(const iMesh& mesh, iRenderGeometry& addTo)
{
    for(int32_t f = 0; f != mesh.getNumberOf3DFaces(); ++f)
    {
        float pointarray[3 * 3];
        float* ptr;
        ptr = pointarray;
        for(int32_t v = 0; v != 3; ++v)
        {
            GetVertPoint(mesh, f, v, ptr);
        }
        addTo.addPolygon(pointarray, 3 * 3);
    }
}
void
cMeshRenderGeometry::AddMeshEdges(const iMesh& mesh, bool includeInternalEdges, iRenderGeometry& addTo)
{
    for(int32_t f = 0; f != mesh.getNumberOf3DFaces(); ++f)
    {
        for(int32_t v = 0; v != 3; ++v)
        {
            if(includeInternalEdges)
            {
                if(mesh.get3DFaceConnection(f, v) < f) // one side for each internal
                {
                    continue;
                }
            }
            else
            {
                if(mesh.get3DFaceConnection(f, v) != -1)
                {
                    continue;
                }
            }
            float data[6];
            float* ptr = data;
            int32_t nextV = v + 1;
            if(nextV == 3)
            {
                nextV = 0;
            }
            GetVertPoint(mesh, f, v, ptr);
            GetVertPoint(mesh, f, nextV, ptr);
            addTo.addLineSequence(data, 6);
        }
    }
}
void
cMeshRenderGeometry::AddUnobstructedSpaceBoundaries(const iMesh& mesh, const iShape& expansionShape, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    mesh.renderUnobstructedSpaceBoundaries(expansionShape, 0, 0, callBack);
}
void
cMeshRenderGeometry::AddSplitBoundaries(const iMesh& mesh, const iShape& expansionShape, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    mesh.renderSplitBoundaries(expansionShape, 0, 0, callBack);
}
void
cMeshRenderGeometry::AddUnsplitBoundaries(const iMesh& mesh, const iShape& expansionShape, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    mesh.renderUnsplitBoundaries(expansionShape, 0, 0, callBack);
}
void
cMeshRenderGeometry::Add3DCross(const iMesh& mesh, cPosition position, float size, iRenderGeometry& addTo)
{
    float x = static_cast<float>(position.x);
    float y = static_cast<float>(position.y);
    float z = mesh.heightAtPositionF(position);
    float coords[6];
    coords[0] = x - size;
    coords[1] = y;
    coords[2] = z;
    coords[3] = x + size;
    coords[4] = y;
    coords[5] = z;
    addTo.addLineSequence(coords, 6);
    coords[0] = x;
    coords[1] = y - size;
    coords[2] = z;
    coords[3] = x;
    coords[4] = y + size;
    coords[5] = z;
    addTo.addLineSequence(coords, 6);
    coords[0] = x;
    coords[1] = y;
    coords[2] = z - size;
    coords[3] = x;
    coords[4] = y;
    coords[5] = z + size;
    addTo.addLineSequence(coords, 6);
}
void
cMeshRenderGeometry::AddLineOnGround(const iMesh& mesh, cPosition start, int32_t endX, int32_t endY, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    mesh.renderLineOnGround(start, endX, endY, 0, 0, callBack);
}
void
cMeshRenderGeometry::AddPath(const iPath& path, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    path.renderOnGround(0, 0, callBack);
}
void
cMeshRenderGeometry::AddRangeBounds(const iMesh& mesh, cHorizontalRange range, iRenderGeometry& addTo)
{
    cRenderGeometryRenderCallBack callBack(addTo);
    mesh.renderRangeBounds(range, 0, 0, callBack);
}

void
cMeshRenderGeometry::Draw3DCross(iTestBed& testBed, const iMesh& mesh, cPosition position, float size)
{
    if(position.cell == -1)
        return;
    auto rg = testBed.newRenderGeometry();
    Add3DCross(mesh, position, size, *rg);
    testBed.render(*rg);
}
void
cMeshRenderGeometry::DrawLineOnGround(iTestBed& testBed, const iMesh& mesh, cPosition start, int32_t endX, int32_t endY)
{
    if(start.cell == -1)
        return;
    auto rg = testBed.newRenderGeometry();
    AddLineOnGround(mesh, start, endX, endY, *rg);
    testBed.render(*rg);
}
void
cMeshRenderGeometry::DrawPath(iTestBed& testBed, const iPath& path)
{
    auto rg = testBed.newRenderGeometry();
    AddPath(path, *rg);
    testBed.render(*rg);
}

void
cMeshRenderGeometry::constructorCommon(iTestBed& testBed, const iMesh& mesh)
{
    _tris = testBed.newRenderGeometry();
    _edges = testBed.newRenderGeometry();
    _baseObstacles = testBed.newRenderGeometry();
    AddMeshTris(mesh, *_tris);
    AddMeshEdges(mesh, false, *_edges);
    cRenderGeometryRenderCallBack callBack(*_baseObstacles);
    mesh.renderBaseObstacles(0, 0, callBack);
}
cMeshRenderGeometry::cMeshRenderGeometry(iTestBed& testBed, const iMesh& mesh)
{
    constructorCommon(testBed, mesh);
}
cMeshRenderGeometry::cMeshRenderGeometry(iTestBed& testBed, const iMesh& mesh, const iShape& expansionShape)
{
    constructorCommon(testBed, mesh);
    _shapeExpansion = testBed.newRenderGeometry();
    AddUnobstructedSpaceBoundaries(mesh, expansionShape, *_shapeExpansion);
}

void
cMeshRenderGeometry::render(iTestBed& testBed) const
{
    testBed.setColourRGB(0.0f, 0.0f, 0.85f);
    testBed.render(*_tris);
    testBed.setColour("blue");
    testBed.render(*_edges);
    testBed.setColour("white");
    testBed.render(*_baseObstacles);
    if(_shapeExpansion)
    {
        testBed.setColour("brown");
        testBed.render(*_shapeExpansion);
    }
}
