
#include "sampleShared/RunPathStressWithConfig.h"
#include "base/types/Error.h"
#include "base/types/Assert.h"
#include "sampleShared/PathStressSetup.h"
#include "sampleShared/MeshRenderGeometry.h"
#include "sampleShared/AgentRenderGeometry.h"
#include "sampleShared/ZoomExtents.h"
#include "sampleShared/LoadBinary.h"
#include "common/STL_Helper.h"
#include <string>
#include <memory>

using std::unique_ptr;
using PE::vector;

static void
CheckPath(iMesh& mesh, iShape& agentShape, iCollisionContext* context, iPath& path)
{
    for(int32_t i = 0; i + 1 < path.size(); i++)
    {
        cPosition segmentStart = path.position(i);
        cPosition segmentEnd = path.position(i + 1);
        assertR(segmentStart != segmentEnd);
        assertR(mesh.testLineCollision(agentShape, context, segmentStart, segmentEnd) == false);
        assertR(mesh.testLineCollision(agentShape, context, segmentEnd, segmentStart) == false);
        {
            int32_t endCell;
            bool collided = mesh.testLineCollision_XY(agentShape, context, segmentStart, segmentEnd.x, segmentEnd.y, endCell);
            assertR(!collided && endCell == segmentEnd.cell);
            cCollidingLine collidingLine;
            unique_ptr<iAgent> agent;
            bool collided2 = mesh.firstCollision(agentShape, context, segmentStart, segmentEnd.x, segmentEnd.y, endCell, collidingLine, agent);
            assertR(!collided2 && endCell == segmentEnd.cell);
        }
    }
}

static void 
DoPathStress( 
    iMesh* mesh,
    iShape& agentShape,
    iCollisionContext* context,
    const cPosition& start, const cPosition& goal,
    const iPathStressVisualisationCallBack* visualisationCallBack

)
{
    auto path = mesh->findShortestPath(agentShape, context, start, goal);
    auto path2 = mesh->findShortestPath6(agentShape, context, start, goal, nullptr);
    if(mesh->testPointCollision(agentShape, context, goal))
    {
        assertR(path == nullptr);
        assertR(path2 == nullptr);
        assertR(mesh->testLineCollision(agentShape, context, start, goal) == true);
        {
            int32_t endCell;
            bool collided = mesh->testLineCollision_XY(agentShape, context, start, goal.x, goal.y, endCell);
            assertR(collided || endCell != goal.cell);
            int32_t endCell2;
            cCollidingLine collidingLine;
            unique_ptr<iAgent> agent;
            bool collided2 = mesh->firstCollision(agentShape, context, start, goal.x, goal.y, endCell2, collidingLine, agent);
            assertR(collided == collided2);
            if(!collided)
            {
                assertR(endCell2 == endCell);
            }
        }
        return;
    }

    assertR((path == nullptr) == (path2 == nullptr));

    if(!path)
    {
        return;
    }

    CheckPath(*mesh, agentShape, context, *path);
    CheckPath(*mesh, agentShape, context, *path2);
    //... compare the two paths, also?

    if(visualisationCallBack)
        visualisationCallBack->renderPath(*path);
}

static void 
DoPathIterations(
    iMesh* mesh, 
    iShape& agentShape,
    iCollisionContext* context,
    int32_t pathIterations,
    const iPathStressVisualisationCallBack* visualisationCallBack
)
{
    do
    {
        if(visualisationCallBack)
            visualisationCallBack->renderMeshAndContext();

        for(int32_t i = 0; i < 10; i++)
        {
            if(!pathIterations)
            {
                break;
            }
            cPosition start;
            do
            {
                start = mesh->generateRandomPosition();
            }
            while(start.cell == -1 || mesh->testPointCollision(agentShape, context, start));

            cPosition goal;
            do
            {
                goal = mesh->generateRandomPosition();
            }
            while(goal.cell == -1);

            DoPathStress(mesh, agentShape, context, start, goal, visualisationCallBack);
            pathIterations--;
        }

        if(visualisationCallBack)
            visualisationCallBack->update();
    }
    while(pathIterations);
}

static unique_ptr<iCollisionContext>
InitContext(iMesh* mesh, const vector<iShape*>& shapes, vector<unique_ptr<iAgent>>& agents)
{
    auto result = mesh->newContext();
    cPosition randomPosition;
    do
    {
        randomPosition = mesh->generateRandomPosition();
    }
    while(randomPosition.cell == -1);
    for(int32_t i = 0; i < SizeL(shapes); i++)
    {
        auto placed = mesh->placeAgent(*shapes[i], randomPosition);
        result->addAgent(*placed);
        agents[i] = std::move(placed);
    }
    return result;
}

static void 
DoDynamicObstacleIterations(
    iMesh* mesh, 
    iShape& agentShape,
    const vector<iShape*>& dynamicObstacleShapes,
    int32_t dynamicObstacleIterations,
    int32_t pathIterations,
    iPathStressVisualisationCallBack* visualisationCallBack
)
{
    int32_t dynamicObstacles = SizeL(dynamicObstacleShapes);
    vector<cPosition> positions(dynamicObstacles);
    vector<unique_ptr<iAgent>> agents(dynamicObstacles);
    unique_ptr<iCollisionContext> context(InitContext(mesh, dynamicObstacleShapes, agents));
    for(int32_t i = 0; i < dynamicObstacleIterations; i++)
    {
        for(int32_t j = 0; j < dynamicObstacles; j++)
        {
            cPosition randomPosition;
            do
            {
                randomPosition = mesh->generateRandomPosition();
            }
            while(randomPosition.cell == -1);
            positions[j] = randomPosition; // the positions are recorded for debugging purposes
            agents[j]->moveTo(randomPosition);
        }
        if(visualisationCallBack)
            visualisationCallBack->newDynamicObstacleSet(*context);
        DoPathIterations(mesh, agentShape, context.get(), pathIterations, visualisationCallBack);
    }
}

void RunPathStressWithConfig(
    iPathEngine* pathengine,
    const std::string& meshName,
    iShape& agentShape,
    const vector<iShape*>& baseObstacleShapes,
    const vector<iShape*>& dynamicObstacleShapes,
    int32_t baseObstacleIterations,
    int32_t dynamicObstacleIterations,
    int32_t pathIterations,
    const char *const* unobstructedSpaceAttributes,
    const char *const* loadOptions,
    iPathStressVisualisationCallBack* visualisationCallBack
)
{
    PE::vector<char> meshBuffer;
    {
        std::string meshPath = "../resource/meshes/" + meshName + ".tok";
        LoadBinary(meshPath.c_str(), meshBuffer);
    }

    int32_t baseObstacles = SizeL(baseObstacleShapes);
    vector<cPosition> positions(baseObstacles);
    for(int32_t i = 0; i < baseObstacleIterations; i++)
    {
        auto mesh = pathengine->loadMeshFromBuffer("tok", VectorBuffer(meshBuffer), SizeU(meshBuffer), loadOptions);

        {
            unique_ptr<iCollisionContext> context(mesh->newContext());
            for(int32_t j = 0; j < baseObstacles; j++)
            {
                cPosition randomPosition;
                do
                {
                    randomPosition = mesh->generateRandomPosition();
                } while(randomPosition.cell == -1);
                unique_ptr<iAgent> agent(mesh->placeAgent(*baseObstacleShapes[j], randomPosition));
                context->addAgent(*agent);
                positions[j] = randomPosition; // recorded for debugging purposes
            }
            mesh->burnContextIntoMesh(*context);
        }

        if(visualisationCallBack)
            visualisationCallBack->newMesh(*mesh, i == 0);

        mesh->generateUnobstructedSpaceFor(agentShape, true, unobstructedSpaceAttributes);
        mesh->generatePathfindPreprocessFor(agentShape, 0);
        {
            //const char* options[] = {
            //    "savePathfindMesh", "true",
            //    "removeExternalFaces", "false",
            //    nullptr
            //};
            //mesh->generatePathfindPreprocess6For(agentShape, options);
            mesh->generatePathfindPreprocess6For(agentShape, nullptr);
        }

        DoDynamicObstacleIterations(mesh.get(), agentShape, dynamicObstacleShapes, dynamicObstacleIterations, pathIterations, visualisationCallBack);

        if(mesh->hasRefs())
        {
            Error("NonFatal", "Outstanding references to iMesh object at end of base obstacle iteration.");
        }
    }
}
